package main

import (
	"encoding/json"
	"log"
	"strconv"
	"strings"

	"github.com/google/go-cmp/cmp"
)

const ignore string = "[NO SYNC]"

func deserialize(body []byte) (map[string]interface{}, error) {
	var model map[string]interface{}
	err := json.Unmarshal(body, &model)
	if err != nil {
		return nil, err
	}
	return model, nil
}

func serialize(model map[string]interface{}) ([]byte, error) {
	body, err := json.Marshal(model)
	if err != nil {
		return nil, err
	}
	return body, nil
}

func matches(someValue interface{}, pred func(string) bool) bool {
	switch value := someValue.(type) {
	case string:
		return pred(value)
	case []interface{}:
		return matchesVec(value, pred)
	case map[string]interface{}:
		return matchesMap(value, pred)
	}
	return false
}

func matchesMap(value map[string]interface{}, pred func(string) bool) bool {
	for _, v := range value {
		if matches(v, pred) {
			return true
		}
	}
	return false
}

func matchesVec(value []interface{}, pred func(string) bool) bool {
	for v := range value {
		if matches(v, pred) {
			return true
		}
	}
	return false
}

func isIgnored(instance *Instance, model map[string]interface{}) bool {
	if instance.SkipPre && strings.HasPrefix(model["id"].(string), "pre-") {
		return true
	}
	return matches(model, func(s string) bool {
		return strings.Contains(s, ignore)
	})
}

func apply(someValue interface{}, path string, fn func(string, string) string) interface{} {
	switch value := someValue.(type) {
	case string:
		return fn(path, value)
	case []interface{}:
		return applyVec(value, path, fn)
	case map[string]interface{}:
		return applyMap(value, path, fn)
	default:
		return value
	}
}

func applyMap(model map[string]interface{}, path string, fn func(string, string) string) map[string]interface{} {
	patched := make(map[string]interface{})
	for k, v := range model {
		patched[k] = apply(v, path+"."+k, fn)
	}
	return patched
}

func applyVec(list []interface{}, path string, fn func(string, string) string) []interface{} {
	patched := make([]interface{}, len(list))
	for i, elem := range list {
		patched[i] = apply(elem, path+"["+strconv.Itoa(i)+"]", fn)
	}
	return patched
}

func isEqualLogDiff(instance *Instance, entityType string, patched map[string]interface{}, target map[string]interface{}) bool {
	diff := cmp.Diff(target, patched)
	eq := diff == ""
	if !eq {
		log.Print(instance.Name + " " + entityType + " " + target["id"].(string) + " has diff:\n" + diff)
	}
	return eq
}
