#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/prometheus/prometheus.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/encode/text/text.h>

#include <util/stream/file.h>
#include <util/stream/format.h>
#include <util/stream/output.h>
#include <util/system/unaligned_mem.h>

#include <array>

using namespace NMonitoring;

class TCliCommand {
public:
    TCliCommand()
        : Opts_(NLastGetopt::TOpts::Default())
    {
        Opts_.AddLongOption('o', "out", "write output into specified file")
            .RequiredArgument("FILE")
            .Optional();
        Opts_.AddHelpOption();
        Opts_.SetFreeArgsNum(2);
        Opts_.SetFreeArgTitle(0, "<file>", "filename or '-' to read from stdin");
    }

    virtual ~TCliCommand() {
    }

    virtual TStringBuf Name() const noexcept = 0;
    virtual TStringBuf Help() const noexcept = 0;

    int Run(int argc, const char* argv[]) {
        Args_.Reset(new NLastGetopt::TOptsParseResult(&Opts_, argc, argv));
        return DoRun();
    }

protected:
    virtual int DoRun() = 0;

    TString ReadInputData() {
        auto freeArgs = Args_->GetFreeArgs();
        TString filename = freeArgs[1];
        if (filename == TStringBuf("-")) {
            return Cin.ReadAll();
        } else {
            return TUnbufferedFileInput(filename).ReadAll();
        }
    }

    IOutputStream* GetOutput() {
        if (Args_->Has("out")) {
            Output_.Reset(new TUnbufferedFileOutput(Args_->Get("out")));
            return Output_.Get();
        }
        return &Cout;
    }

protected:
    NLastGetopt::TOpts Opts_;
    THolder<NLastGetopt::TOptsParseResult> Args_;
    THolder<IOutputStream> Output_;
};

class TFromJsonCmd: public TCliCommand {
public:
    TFromJsonCmd() {
        Opts_.AddLongOption(
                'c', "compress",
                "use specified compression algorithm when writing in spack. "
                "Possible values are: { IDENTITY, ZLIB, ZSTD, LZ4 }")
            .Optional()
            .RequiredArgument("ALG")
            .DefaultValue("ZSTD");
    }

private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("from-json");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from json into spack");
    }

    int DoRun() override {
        auto encoder = EncoderSpackV1(
                GetOutput(),
                ETimePrecision::SECONDS,
                FromString<ECompression>(Args_->Get("compress")));
        TString data = ReadInputData();
        DecodeJson(data, encoder.Get());
        return 0;
    }
};

class TToJsonCmd: public TCliCommand {
public:
    TToJsonCmd() {
        Opts_.AddLongOption("pretty", "print pretty json").NoArgument();
    }

private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("to-json");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from spack into json");
    }

    int DoRun() override {
        auto encoder = EncoderJson(GetOutput(), Args_->Has("pretty"));
        TString data = ReadInputData();
        TMemoryInput mem(data);
        DecodeSpackV1(&mem, encoder.Get());
        return 0;
    }
};

class TToTextCmd: public TCliCommand {
private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("to-text");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from spack into text");
    }

    int DoRun() override {
        auto encoder = EncoderText(GetOutput());
        TString data = ReadInputData();
        TMemoryInput mem(data);
        DecodeSpackV1(&mem, encoder.Get());
        return 0;
    }
};

class TFromPromCmd: public TCliCommand {
public:
    TFromPromCmd() {
        Opts_.AddLongOption(
                'c', "compress",
                "use specified compression algorithm when writing in spack. "
                "Possible values are: { IDENTITY, ZLIB, ZSTD, LZ4 }")
            .Optional()
            .RequiredArgument("ALG")
            .DefaultValue("ZSTD");
    }

private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("from-prom");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from prometheus text format into spack");
    }

    int DoRun() override {
        auto encoder = EncoderSpackV1(
                GetOutput(),
                ETimePrecision::SECONDS,
                FromString<ECompression>(Args_->Get("compress")));
        TString data = ReadInputData();
        DecodePrometheus(data, encoder.Get());
        return 0;
    }
};

class TToPromCmd: public TCliCommand {
private:
    TStringBuf Name() const noexcept override {
        return TStringBuf("to-prom");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("converts metrics from spack into prometheus text format");
    }

    int DoRun() override {
        auto encoder = EncoderPrometheus(GetOutput());
        TString data = ReadInputData();
        TMemoryInput mem(data);
        DecodeSpackV1(&mem, encoder.Get());
        return 0;
    }
};

class THeaderCmd: public TCliCommand {
    TStringBuf Name() const noexcept override {
        return TStringBuf("header");
    }

    TStringBuf Help() const noexcept override {
        return TStringBuf("prints information stored in spack header");
    }

    int DoRun() override {
        TString data = ReadInputData();
        if (data.size() < sizeof(TSpackHeader)) {
            Cerr << "too small input file" << Endl;
            std::exit(1);
        }

        auto h = ReadUnaligned<TSpackHeader>(data.data());
        if (h.Magic != 0x5053) {
            Cerr << "invalid file format" << Endl;
            return 1;
        }

        Cout << "            magic: " << Hex(h.Magic) << '\n'
             << "          version: " << (h.Version >> 8) << '.' << (h.Version & 0xff) << '\n'
             << "      header size: " << h.HeaderSize << '\n'
             << "   time precision: " << DecodeTimePrecision(h.TimePrecision) << '\n'
             << "      compression: " << DecodeCompression(h.Compression) << '\n'
             << " label names size: " << h.LabelNamesSize << '\n'
             << "label values size: " << h.LabelValuesSize << '\n'
             << "     metric count: " << h.MetricCount << '\n'
             << "     points count: " << h.PointsCount << '\n'
             << Flush;
        return 0;
    }
};

std::array<TCliCommand*, 6> Commands = {{
    Singleton<TFromJsonCmd>(),
    Singleton<TToJsonCmd>(),
    Singleton<TFromPromCmd>(),
    Singleton<TToPromCmd>(),
    Singleton<TToTextCmd>(),
    Singleton<THeaderCmd>(),
}};

void PrintUsage(const char* appName) {
    size_t maxLen = 0;
    for (const TCliCommand* cmd: Commands) {
        maxLen = Max(maxLen, cmd->Name().size());
    }

    Cerr << appName << " [-h] <command> [OPTIONS]\n\n"
         << "Known commands are:\n";

    for (const TCliCommand* cmd: Commands) {
        Cerr << "  * " << RightPad(cmd->Name(), maxLen, ' ')
             << "    " << cmd->Help() << '\n';
    }

    Cerr << Flush;
}

int main(int argc, const char* argv[]) {
    try {
        if (argc <= 1) {
            PrintUsage(argv[0]);
            return 1;
        }

        TStringBuf cmdName = argv[1];
        if (cmdName == TStringBuf("-h") || cmdName == TStringBuf("--help")) {
            PrintUsage(argv[0]);
            return 0;
        }

        for (TCliCommand* cmd: Commands) {
            if (cmdName == cmd->Name()) {
                return cmd->Run(argc, argv);
            }
        }

        Cerr << "Unknown command: '" << cmdName << '\'' << Endl;
        PrintUsage(argv[0]);
        return 1;
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}
