#include "buffer.h"
#include "log.h"

#include <util/digest/city.h>

#define DEFAULT_CONTAINER_SIZE 1024


inline void *xxmalloc(size_t size, const char *f) {
    void *m;
    if ((m = malloc(size)) == NULL)
        Fatal("ERROR: Cannot allocate %lu bytes in %s().\n", size, f);
    return m;
}
#define xmalloc(X) xxmalloc(X, __func__)


inline void *xxzalloc(size_t size, const char *f) {
    return memset(xxmalloc(size, f), 0, size);
}
#define xzalloc(X) xxzalloc(X, __func__)


inline void *xxrealloc(void *ptr, size_t size, const char *f) {
    void *m;
    if ((m = realloc(ptr, size)) == NULL) {
        free(ptr);
        Fatal("ERROR: Cannot reallocate %lu bytes in %s().\n", size, f);
    }
    return m;
}
#define xrealloc(X, Y) xxrealloc(X, Y, __func__)


TMyBuffer::TMyBuffer() {
    Container = xmalloc(DEFAULT_CONTAINER_SIZE);
    ContainerSize = DEFAULT_CONTAINER_SIZE;
}

TMyBuffer::TMyBuffer(const char *c) {
    Container = xmalloc(DEFAULT_CONTAINER_SIZE);
    ContainerSize = DEFAULT_CONTAINER_SIZE;
    *this += c;
}

TMyBuffer::~TMyBuffer() {
    if (Container != nullptr) {
        free(Container);
        Container     = nullptr;
        ContainerSize = 0;
        DataSize      = 0;
    }
}

size_t TMyBuffer::size() {
    return DataSize;
}

char *TMyBuffer::data() {
    return (char *)Container;
}

void TMyBuffer::resize(size_t NewSize) {
    DataSize = NewSize;
    if (NewSize + 1 > ContainerSize) {
        NewSize += 1;
        for (ContainerSize = 1; NewSize != 0; NewSize >>= 1, ContainerSize <<= 1);
        Container = xrealloc(Container, ContainerSize);
    }
    else if ((NewSize << 1) <= ContainerSize && NewSize > DEFAULT_CONTAINER_SIZE) {
        for (ContainerSize = 1; NewSize != 0; NewSize >>= 1, ContainerSize <<= 1);
        Container = xrealloc(Container, ContainerSize);
    }
    *((char *)Container + DataSize) = '\0';
}

void TMyBuffer::zresize(size_t NewSize) {
    size_t From = DataSize;
    resize(NewSize);
    bzero((char *)Container + From, DataSize - From);
}

void TMyBuffer::append(const char *Str, size_t Length) {
    resize(DataSize + Length);
    memcpy((char *)Container + DataSize - Length, Str, Length);
}

void TMyBuffer::append(size_t Size, char c) {
    resize(DataSize + Size);
    memset((char *)Container + DataSize - Size, c, Size);
}

void TMyBuffer::zappend(size_t Size) {
    resize(DataSize + Size);
    bzero((char *)Container + DataSize - Size, Size);
}

void TMyBuffer::clear() {
    resize(DEFAULT_CONTAINER_SIZE);
    *(char *)Container = '\0';
    DataSize = 0;
}

void TMyBuffer::erase(size_t From, size_t Length) {
    if (From >= DataSize)
        return;
    // Change of sense: move bytes To<-From:Length
    size_t To = From;
    From += Length;
    if (From >= DataSize) {
        resize(To);
        return;
    }
    Length = DataSize - From;
    memmove((char *)Container + To, (char *)Container + From, Length);
    resize(To + Length);
}

void *TMyBuffer::begin() {
    return Container;
}

void *TMyBuffer::end() {
    return (void *)((char *)Container + DataSize);
}

char &TMyBuffer::front() {
    return *(char *)Container;
}

char &TMyBuffer::back() {
    if (DataSize == 0)
        return *(char *)Container;
    return *((char *)Container + DataSize - 1);
}

char *TMyBuffer::substr(size_t Begin, size_t Length) {
    size_t S = DataSize;
    resize(DataSize + Length + 1);
    DataSize = S;
    memcpy((char *)Container + DataSize + 1, (char *)Container + Begin, Length);
    return (char *)Container + DataSize + 1;
}

size_t TMyBuffer::hash() {
    return CityHash64(reinterpret_cast<const char*>(Container), ContainerSize);
}

size_t TMyBuffer::subhash(size_t Start, size_t Length) const {
    return CityHash64(reinterpret_cast<const char*>(Container) + Start, Length);
}

char &TMyBuffer::operator[](size_t idx) {
    return *((char *)Container + idx);
}

const char &TMyBuffer::operator[](size_t idx) const {
    return *((char *)Container + idx);
}

TMyBuffer::TMyBuffer(const TMyBuffer &other) {
    *this = other;
}

TMyBuffer &TMyBuffer::operator=(const TMyBuffer &other) {
    if (this != &other) {
        this->Container = xrealloc(this->Container, other.ContainerSize);
        memcpy(this->Container, other.Container, other.DataSize);
        this->ContainerSize = other.ContainerSize;
        this->DataSize = other.DataSize;
    }
    return *this;
}

TMyBuffer &TMyBuffer::operator=(const std::string &str) {
    this->resize(str.size());
    memcpy(this->Container, str.c_str(), str.size());
    return *this;
}

TMyBuffer &TMyBuffer::operator=(const char *c) {
    const char *p = c;
    for (; *p != 0; ++p);
    this->resize(p - c);
    memcpy(this->Container, c, p - c);
    return *this;
}

TMyBuffer::TMyBuffer(TMyBuffer &&other) {
    *this = std::move(other);
}

TMyBuffer &TMyBuffer::operator=(TMyBuffer&& other) noexcept { // move assignment
    if (this != &other) {
        void *p  = this->Container; this->Container     = other.Container;     other.Container     = p;
        size_t s = this->DataSize;  this->DataSize      = other.DataSize;      other.DataSize      = s;
        s = this->ContainerSize;    this->ContainerSize = other.ContainerSize; other.ContainerSize = s;
    }
    return *this;
}

TMyBuffer &TMyBuffer::operator+=(const TMyBuffer &r) {
    size_t NewSize = this->DataSize + r.DataSize;
    this->resize(NewSize);
    memcpy((char *)this->Container + this->DataSize, r.Container, r.DataSize);
    this->DataSize = NewSize;
    return *this;
}

TMyBuffer &TMyBuffer::operator+=(const std::string &r) {
    size_t NewSize = this->DataSize + r.size();
    this->resize(NewSize);
    memcpy((char *)this->Container + this->DataSize - r.size(), r.data(), r.size());
    return *this;
}

TMyBuffer &TMyBuffer::operator+=(const char *c) {
    const char *p = c;
    size_t Size;
    for (; *p != 0; ++p);
    Size = p - c;
    this->resize(this->DataSize + Size);
    memcpy((char *)this->Container + this->DataSize - Size, c, Size);
    return *this;
}

TMyBuffer operator+(TMyBuffer l, const TMyBuffer &r) {
    l += r;
    return l;
}

TMyBuffer operator+(TMyBuffer l, const std::string &r) {
    l += r;
    return l;
}

TMyBuffer operator+(TMyBuffer l, const char *r) {
    l += r;
    return l;
}

bool operator<(const TMyBuffer &l, const TMyBuffer &r) {
    if (r.DataSize == 0)
        return false;
    else if (l.DataSize == 0)
        return true;
    size_t MinSize = (l.DataSize < r.DataSize) ? l.DataSize : r.DataSize;
    return memcmp(l.Container, r.Container, MinSize) < 0;
}

bool operator==(const TMyBuffer &l, const TMyBuffer &r) {
    if (r.DataSize != l.DataSize)
        return false;
    return memcmp(l.Container, r.Container, r.DataSize) == 0;
}

inline bool operator!=(const TMyBuffer &l, const TMyBuffer &r) { return !(l == r); }
inline bool operator> (const TMyBuffer &l, const TMyBuffer &r) { return r < l; }
inline bool operator<=(const TMyBuffer &l, const TMyBuffer &r) { return !(l > r); }
inline bool operator>=(const TMyBuffer &l, const TMyBuffer &r) { return !(l < r); }
