#include <unistd.h>
#include <string>
#include <iostream>
#include <stdexcept>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <string.h>

#include "classip.h"
#include "log.h"

TIP::TIP(std::string N, uint16_t P) {
    ConstructFromNamePort(N, P);
}

#define BUFLEN 8
void TIP::ConstructFromNamePort(std::string N, uint16_t P) {
    Name = N;
    Port = P;

    char Buf[BUFLEN];
    int gaiResult;
    struct addrinfo *AddressInfo = NULL;
    struct addrinfo hints;

    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_family = AF_UNSPEC;    // Allow IPv4 or IPv6
    hints.ai_flags = 0;
    hints.ai_protocol = 0;          // Any protocol

    snprintf(Buf, BUFLEN, "%u", Port);
    gaiResult = getaddrinfo(Name.c_str(), Buf, &hints, &AddressInfo);
    if (gaiResult != 0)
        Fatal("getaddrinfo (%i): %s '%s'", gaiResult, gai_strerror(gaiResult), Name.c_str());
    if (AddressInfo != NULL) {
        Family = AddressInfo->ai_family;
        if (Family == AF_INET) {
            memcpy(&SockAddress, AddressInfo->ai_addr, sizeof(struct sockaddr_in));
            inet_ntop(SockAddress.sin_family, &SockAddress.sin_addr, IPChar, INET_ADDRSTRLEN);
            Address = SockAddress.sin_addr.s_addr;
        } else {
            memcpy(&SockAddress6, AddressInfo->ai_addr, sizeof(struct sockaddr_in6));
            inet_ntop(SockAddress6.sin6_family, &SockAddress6.sin6_addr, IPChar, INET6_ADDRSTRLEN);
            memcpy(Address6, &SockAddress6.sin6_addr.s6_addr, sizeof(Address6));
        }
        freeaddrinfo(AddressInfo);
    }
}

TIP::TIP(std::string Str) {
    ConstructFromHostPort(Str);
}
void TIP::ConstructFromHostPort(std::string Str) {
    size_t Begin, End;
    Port = 0;

    Begin = Str.find("[");
    if (Begin != std::string::npos) {
        End = Str.find("]");
        if (End == std::string::npos)
            Fatal("Bad Host:Port in '%s'", Str.c_str());
        Name = Str.substr(Begin + 1, End - 1);
        End = Str.find(":", End);
    }
    else {
        End = Str.find(":");
        Name = Str.substr(0, End);
    }
    if (End != std::string::npos) {
        try {
            Port = std::stoi(Str.substr(End + 1));
        }
        catch (std::invalid_argument) {
            Fatal("Bad Host:Port in '%s'", Str.c_str());
        }
    }
    ConstructFromNamePort(Name, Port);
}

TIP::TIP(uint32_t Addr, uint16_t P) {
    ConstructFromAddrPort(Addr, P);
}
void TIP::ConstructFromAddrPort(uint32_t Addr, uint16_t P) {
    Address = Addr;
    Port = P;
    Family = AF_INET;
    inet_ntop(Family, &Address, IPChar, INET_ADDRSTRLEN);
    SockAddress.sin_family = AF_INET;
    SockAddress.sin_port = htons(P);
    SockAddress.sin_addr.s_addr = Address;
}

TIP::TIP(uint8_t *Addr6, uint16_t P) {
    ConstructFromAddr6Port(Addr6, P);
}
void TIP::ConstructFromAddr6Port(uint8_t *Addr6, uint16_t P) {
    memcpy(Address6, Addr6, sizeof(Address6));
    Port = P;
    Family = AF_INET6;
    inet_ntop(Family, &Address6, IPChar, INET6_ADDRSTRLEN);
    SockAddress6.sin6_family = AF_INET6;
    SockAddress6.sin6_port = htons(P);
    //SockAddress6.sin6_flowinfo = RANDOM;
    SockAddress6.sin6_scope_id = 0;
    memcpy(&SockAddress6.sin6_addr.s6_addr, Addr6, sizeof(Address6));
}

TIP::TIP(struct sockaddr *SockAddr) {
    ConstructFromSockaddr(SockAddr);
}
void TIP::ConstructFromSockaddr(struct sockaddr *SockAddr) {
    if (SockAddr->sa_family == AF_INET)
        ConstructFromSockaddrIn((struct sockaddr_in *)SockAddr);
    else
        ConstructFromSockaddr6In((struct sockaddr_in6 *)SockAddr);
}

TIP::TIP(struct sockaddr_in *SockAddr) {
    ConstructFromSockaddrIn(SockAddr);
}
void TIP::ConstructFromSockaddrIn(struct sockaddr_in *SockAddr) {
    Family = AF_INET;
    Address = SockAddr->sin_addr.s_addr;
    Port = ntohs(SockAddr->sin_port);
    memcpy(&SockAddress, SockAddr, sizeof(struct sockaddr_in));
    inet_ntop(AF_INET, &SockAddress.sin_addr, IPChar, INET_ADDRSTRLEN);
}

TIP::TIP(struct sockaddr_in6 *SockAddr6) {
    ConstructFromSockaddr6In(SockAddr6);
}
void TIP::ConstructFromSockaddr6In(struct sockaddr_in6 *SockAddr6) {
    Family = AF_INET6;
    memcpy(Address6, &SockAddr6->sin6_addr.s6_addr, sizeof(Address6));
    Port = ntohs(SockAddr6->sin6_port);
    memcpy(&SockAddress6, SockAddr6, sizeof(struct sockaddr_in6));
    inet_ntop(AF_INET6, &SockAddress6.sin6_addr, IPChar, INET6_ADDRSTRLEN);
}

size_t TIP::hash() const {
    size_t h = Port;
    if (Family == AF_INET) {
        h = (h << 32) + Address;
    }
    else
        h = (h << 32) + *((uint64_t *)Address6) + *((uint64_t *)Address6 + 1);
    return h;
}

TIP &TIP::operator=(const TIP &r) {
    this->Name = r.Name;
    this->Port = r.Port;
    this->Address = r.Address;
    this->Family = r.Family;
    memcpy(&this->Address6, &r.Address6, 16);
    memcpy(&this->IPChar, &r.IPChar, INET6_ADDRSTRLEN + 1);
    memcpy(&this->SockAddress, &r.SockAddress, sizeof(struct sockaddr_in));
    memcpy(&this->SockAddress6, &r.SockAddress6, sizeof(struct sockaddr_in6));
    return *this;
}

const char *TIP::GetIPChar() const {
    return IPChar;
}

const std::string TIP::GetName() {
    return Name;
}

const struct sockaddr_in *TIP::GetSockAddr() const {
    return &SockAddress;
}

const struct sockaddr_in6 *TIP::GetSockAddr6() const {
    return &SockAddress6;
}

uint16_t TIP::GetPort() const {
    return Port;
}

int TIP::GetFamily() const {
    return Family;
}

bool operator<(const TIP &a, const TIP &b) {
    if (a.Family == AF_INET6) {
        if (b.Family == AF_INET6) {
            uint64_t *aAddr6_64 = (uint64_t *)a.Address6;
            uint64_t *bAddr6_64 = (uint64_t *)b.Address6;
            if (*aAddr6_64 == *bAddr6_64) {
                ++aAddr6_64;
                ++bAddr6_64;
                if (*aAddr6_64 == *bAddr6_64)
                    return a.Port < b.Port;
                return *aAddr6_64 < *bAddr6_64;
            }
            return *aAddr6_64 < *bAddr6_64;
        }
        return false;
    }
    if (b.Family == AF_INET6)
        return true;
    if (a.Address == b.Address)
        return a.Port < b.Port;
    return a.Address < b.Address;
}

bool operator==(const TIP &a, const TIP &b) {
    if (a.Family != b.Family)
        return false;
    if (a.Family == AF_INET6) {
        uint64_t *aAddr6_64 = (uint64_t *)a.Address6;
        uint64_t *bAddr6_64 = (uint64_t *)b.Address6;
        if (*aAddr6_64 == *bAddr6_64) {
            ++aAddr6_64;
            ++bAddr6_64;
            if (*aAddr6_64 == *bAddr6_64)
                return a.Port == b.Port;
            return false;
        }
        return false;
    }
    return a.Address == b.Address && a.Port == b.Port;
}
