#include <iostream>

#include "log.h"
#include "pcap.h"

TPcap::TPcap() {
    Log("Using %s", pcap_lib_version());
    if (pcap_findalldevs(&alldevs, errbuf) < 0)
        Fatal("ERROR: %s", errbuf);

    pcap_if_t *devp = alldevs;
    while (devp != nullptr) {
#ifdef PCAP_IF_UP
        if (devp->flags & PCAP_IF_UP && devp->flags & PCAP_IF_RUNNING) {
#endif
            IFaceMap[devp->name].isLoopback = (devp->flags & PCAP_IF_LOOPBACK);
            pcap_addr_t *addrp = devp->addresses;
            while (addrp != nullptr) {
                if (addrp->addr->sa_family == AF_INET || addrp->addr->sa_family == AF_INET6)
                    IFaceMap[devp->name].AddressVector.push_back(TIP(addrp->addr));
                addrp = addrp->next;
            }
#ifdef PCAP_IF_UP
        }
#endif
        devp = devp->next;
    }
}

TPcap::~TPcap() {
    if (CaptureHandle != nullptr) {
        LogStats();
        pcap_close(CaptureHandle);
    }
    pcap_freealldevs(alldevs);
}

int TPcap::CheckIface(std::string Interface) {
    for (auto it = IFaceMap.begin(); it != IFaceMap.end(); ++it) {
        if (it->first == Interface)
            return 0;
    }
    return -1;
}

void TPcap::Init(std::string Interface, std::string Filter, int PacketsMaxCount) {
    MaxPackets = PacketsMaxCount;

    if (CheckIface(Interface) < 0)
        Fatal("ERROR: Cannot capture on iface '%s'", Interface.c_str());
    if ((CaptureHandle = pcap_create(Interface.c_str(), errbuf)) == NULL)
        Fatal("ERROR: Cannot create pcap handle: '%s'", errbuf);
    if (pcap_set_snaplen(CaptureHandle, SNAPLEN) < 0)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    if (pcap_set_buffer_size(CaptureHandle, BUFCAPLEN) < 0)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    if (pcap_set_timeout(CaptureHandle, TIMEOUTMS) < 0)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));

    int e = pcap_activate(CaptureHandle);
    if (e < 0)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    else if (e > 0)
        Log("WARNING: %s", pcap_geterr(CaptureHandle));

    if (Filter.length() > 0) {
        Log("Setting filter '%s'", Filter.c_str());
        if (pcap_compile(CaptureHandle, &bpfp, Filter.c_str(), OPTIMIZE_CODE, PCAP_NETMASK_UNKNOWN) < 0)
            Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
        if (pcap_setfilter(CaptureHandle, &bpfp) < 0)
            Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    }
    IfaceLinkType = pcap_datalink(CaptureHandle);
    if (IfaceLinkType == LINKTYPE_ETHERNET)
        Log("Selecting ETHERNET link type");
    else if (IfaceLinkType == LINKTYPE_LINUX_SLL)
        Log("Selecting LINUX_SLL link type");
    else
        Fatal("ERROR: Cannot capture on %i link type", IfaceLinkType);
}

void TPcap::SetNonBlock() {
    if (pcap_setnonblock(CaptureHandle, 1, errbuf) < 0)
        Fatal("ERROR: Cannot set nonblock: '%s'", errbuf);
}

int TPcap::GetFileDescriptor() {
    SelectableFD = pcap_get_selectable_fd(CaptureHandle);
    if (SelectableFD < 0)
        Fatal("ERROR: Cannot get selectable file descriptor");
    return SelectableFD;
}

void TPcap::Loop(pcap_handler callback, u_char *UserData) {
    int e = pcap_loop(CaptureHandle, MaxPackets, callback, UserData);
    if (e == -1)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    else if (e == -2)
        Log("EXIT: loop interrupted");
}

void TPcap::BreakLoop() {
    pcap_breakloop(CaptureHandle);
}

int TPcap::GetPacket(struct pcap_pkthdr **Header, const u_char **Data) {
    if (MaxPackets > 0 && ++PacketCounter >= MaxPackets)
        return -1;

    int Status = pcap_next_ex(CaptureHandle, Header, Data);
    if (Status < 0)
        Fatal("ERROR: %s", pcap_geterr(CaptureHandle));
    return Status;
}

void TPcap::LogStats() {
    struct pcap_stat ps;
    if (pcap_stats(CaptureHandle, &ps) < 0) {
        Log("%s", pcap_geterr(CaptureHandle));
        return;
    }
    Log("%u packets received, %u packets dropped due to small buffer or slow reader, %u packets dropped by the network interface or its driver", ps.ps_recv, ps.ps_drop, ps.ps_ifdrop);
}

void TPcap::IfaceCout() {
    for (auto it = IFaceMap.begin(); it != IFaceMap.end(); ++it) {
        std::cout << it->first << ":" << std::endl;
        for (auto ita = it->second.AddressVector.begin(); ita != it->second.AddressVector.end(); ++ita)
            std::cout << "\t" << ita->GetIPChar() << std::endl;
    }
}

int TPcap::GetIfaceLinkType() {
    return IfaceLinkType;
}
