#!/usr/bin/env python3

import os
import sys
import argparse
import datetime
import time
import json
import ssl
import urllib.request
import urllib.parse
import urllib.error
import subprocess

from sandbox.common import rest
from sandbox.common.auth import OAuth
import library.python.oauth as oauth


# ==================================================================================================================


DefaultKeyRobot = "robot-solomon-build"
ClientID = "350dd730a52a4a018323255d235889f5"
ClientPassword = "0dd1e4d7b381494180255114e2faa990"


# ==================================================================================================================


def Log(String, Time=True):
    TimeStr = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S.%f - ") if Time else ""
    print(TimeStr + String)


class Exec:
    def __init__(self, Command, Env=None, ErrToOut=False, Stdin=None):
        self.pid = -1
        self.out = ""
        self.err = ""
        self.status = 1
        try:
            self.p = subprocess.Popen(
                Command,
                env=Env,
                stdin=subprocess.PIPE if Stdin is True else Stdin,
                stdout=subprocess.PIPE,
                stderr=subprocess.STDOUT if ErrToOut else subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            try:
                self.out = self.p.stdout.read().strip().decode("utf-8")
                self.err = "" if ErrToOut else self.p.stderr.read().strip().decode("utf-8")
            except KeyboardInterrupt:
                os.killpg(self.pid, 2)
            self.p.wait()
            self.status = self.p.returncode
        except OSError as e:
            Log("Got exception running {}: {}".format(Command, e))
            exit(1)


class TTC:
    def __init__(self):
        self.Enabled = False
        self.NameList = []

    def Enable(self):
        self.Enabled = True

    def Disable(self):
        self.Enabled = False

    def __call__(self, Name):
        if self.Enabled:
            print("##teamcity[blockOpened name='{}']".format(Name))
            self.NameList.append(Name)

    def Close(self):
        if self.Enabled and len(self.NameList) > 0:
            print("##teamcity[blockClosed name='{}']".format(self.NameList.pop()))


class GetURL:
    def __init__(self, URL, Headers={}, Data=None, Timeout=10, Verify=False, Method=None):
        Ctx = ssl.create_default_context()
        if not Verify:
            Ctx.check_hostname = False
            Ctx.verify_mode = ssl.CERT_NONE
        self.code = 0
        self.err = ""
        self.text = ""
        try:
            if isinstance(Data, dict):
                Data = urllib.parse.urlencode(Data)
            if isinstance(Data, str):
                Data = Data.encode("ascii")
            Req = urllib.request.Request(URL, data=Data, headers=Headers, method=Method)
            Resp = urllib.request.urlopen(Req, timeout=Timeout, context=Ctx)
            self.text = Resp.read().decode("utf-8")
            self.code = Resp.getcode()
        except urllib.error.HTTPError as e:
            self.code = e.code
            self.text = e.read().decode("utf-8")
            self.err = e.reason
        except urllib.error.URLError as e:
            self.err = e.reason
        except Exception as e:
            self.err = str(e)


def GetBranchLatestRevision(Branch):
    Branch = Branch.strip("/")
    if Branch != "trunk" and not Branch.startswith("branches/"):
        Branch = "branches/{}".format(Branch)
    R = Exec(["/usr/bin/svn", "info", "--show-item", "last-changed-revision", "svn+ssh://arcadia.yandex.ru/arc/{}".format(Branch)])
    if R.status != 0:
        Log("Cannot get branch and last revision ({}): {}".format(R.status, R.err))
        exit(1)
    return Branch, R.out.strip()


# ==================================================================================================================


class TZ2:
    def __init__(self, Endpoint, Cfg, ApiKey):
        self.Endpoint = Endpoint
        self.Cfg = Cfg
        self.ApiKey = ApiKey
        self.RefURL = "https://{}/control_panel?configId={}".format(Endpoint, Cfg)
        self.ApiURL = "https://{}/api/v1".format(Endpoint)

    def Edit(self, PkgDict):
        """PkgDict = {"name": "version", ... }
        """
        URL = "{}/editItems".format(self.ApiURL)
        Data = {
            "configId": self.Cfg,
            "apiKey":   self.ApiKey,
            "items":    json.dumps([{"name": P, "version": V} for P, V in PkgDict.items()], separators=(',', ':'))
        }
        Log("Z2 edit config ({} -> {}):".format(self.Cfg, json.dumps(PkgDict)))
        R = GetURL(URL, Data=Data)
        if R.code != 200:
            Log("Failed to edit items {} (code={}): {}".format(self.RefURL, R.code, R.err))
            return False
        else:
            Log("Success editing {} (code={}): {}".format(self.RefURL, R.code, R.text))
            return True

    def SelectiveEdit(self, PkgDict):
        """PkgDict = {"name": "version", ... }
        """
        PkgSet = {Item for Item in self.ListItems()}
        if len(PkgSet) == 0:
            Log("No packages found in {}".format(self.RefURL))
            return False
        SelectedPkgDict = {K: V for K, V in PkgDict.items() if K in PkgSet}
        if len(SelectedPkgDict) == 0:
            Log("No packages selected for {}".format(self.RefURL))
            return True
        Log("Selected packages for {}: {}".format(self.Cfg, SelectedPkgDict.keys()))
        return self.Edit(SelectedPkgDict)

    def ListItems(self):
        """PkgDict = {"name": "version", ... }
        """
        URL = "{}/items?configId={}&apiKey={}".format(self.ApiURL, self.Cfg, self.ApiKey)
        R = GetURL(URL)
        if R.code != 200:
            Log("Failed to list items {} (code={}): {}".format(self.RefURL, R.code, R.err))
        else:
            try:
                Dict = json.loads(R.text)
                if Dict.get("success"):
                    return {Item["name"]: Item["version"] for Item in Dict.get("response", {}).get("items", [])}
                Log("Failed to list items for {}: {}".format(self.RefURL, Dict.get("errorMsg", "general error")))
            except Exception as e:
                Log("Failed to decode list items for {}: {}".format(self.RefURL, e))
        return {}

    def UpdateStart(self, Retries=2):
        URL = "{}/update".format(self.ApiURL)
        Data = {
            "configId": self.Cfg,
            "apiKey":   self.ApiKey
        }
        while Retries >= 0:
            Log("Z2 run update for {}".format(self.RefURL))
            R = GetURL(URL, Data=Data)
            if R.code != 200:
                Log("Failed to start update {} (code={}): {}".format(self.RefURL, R.code, R.err))
                Retries -= 1
            else:
                Log("Success start update {} (code={}): {}".format(self.RefURL, R.code, R.text))
                return True
            if Retries >= 0:
                time.sleep(5)
        return False

    def CheckUpdate(self):
        URL = "{}/updateStatus?configId={}&apiKey={}".format(self.ApiURL, self.Cfg, self.ApiKey)
        R = GetURL(URL)
        if R.code != 200:
            Log("Failed to check update {} (code={}): {}".format(self.RefURL, R.code, R.err))
        else:
            try:
                Dict = json.loads(R.text)
                if Dict.get("success"):
                    return Dict.get("response", {})
                Log("Failed to get update state {}: {}".format(self.RefURL, Dict.get("errorMsg", "general error")))
            except Exception as e:
                Log("Failed to decode check update for {}: {}".format(self.RefURL, e))
        return {}

    def UpdateWait(self, Timeout=0):
        MaxTime = time.time() + Timeout if Timeout > 0 else 0
        while time.time() < MaxTime if MaxTime > 0 else True:
            Dict = self.CheckUpdate()
            if Dict.get("updateStatus") == "FINISHED":
                if Dict.get("result") == "SUCCESS":
                    Log("Success updating {}".format(self.RefURL))
                    return True
                else:
                    Log("Done {}: {} - failed workers: {}".format(self.RefURL, Dict.get("result"), " ".join(Dict.get("failedWorkers", []))))
                    break
            Log("    {}".format(Dict.get("updateStatus", "bad update status")))
            time.sleep(5)
        else:
            Log("Updating {} timed out ({} sec)".format(self.RefURL, Timeout))
        return False


def Deploy(CfgList,
           PkgDict,
           PushDeploy=True,
           Parallel=False,
           YavToken=None,
           YavSecretId=None,
           ApiKey=None,
           Endpoint=None,
           Retries=2,
           StartRetries=30,
           Timeout=900
           ):
    ApiKeyDict = {}
    TC("Checking Z2 tokens")
    if YavSecretId:
        if not YavToken:
            Log("No yav secret id provided")
            TC.Close()
            return False
        Headers = {"Authorization": "Bearer {}".format(YavToken)}
        R = GetURL("https://vault-api.passport.yandex.net/1/versions/{}".format(YavSecretId), Headers=Headers)
        if R.code != 200:
            Log("Failed to get yav secret {} (code={}): {}".format(YavSecretId, R.code, R.text))
            TC.Close()
            return False
        ApiKeyDict = {D.get("key"): D.get("value") for D in json.loads(R.text).get("version", {}).get("value", [])}
    elif not ApiKey:
        Log("No z2 api key, no yav token and no yav secret provided")
        TC.Close()
        return False
    TC.Close()

    ZList = []
    for Cfg in CfgList:
        E = Endpoint if Endpoint else "z2-cloud.yandex-team.ru" if "_CLOUD_" in Cfg else "z2.yandex-team.ru"
        K = ApiKeyDict.get(Cfg, ApiKey)
        if not K:
            Log("Failed to find Z2 API key for {}".format(Cfg))
            return False
        ZList.append(TZ2(E, Cfg, K))
        TC("Editing {}".format(Cfg))
        for _ in range(Retries + 1):
            if ZList[-1].SelectiveEdit(PkgDict):
                break
            time.sleep(5)
        else:
            TC.Close()
            return False
        TC.Close()
    if not PushDeploy:
        return True

    if Parallel:
        NewList = []
        for _ in range(Retries + 1):
            GoodList = []
            BadList = []
            for Z in ZList:
                TC("Updating {}".format(Z.Cfg))
                if Z.UpdateStart(Retries=StartRetries):
                    GoodList.append(Z)
                else:
                    BadList.append(Z)
                TC.Close()
            for Z in GoodList:
                TC("Waiting {} deploy".format(Z.Cfg))
                if Z.UpdateWait(Timeout=Timeout):
                    NewList.append(Z)
                else:
                    BadList.append(Z)
                TC.Close()
            if len(BadList) == 0:
                break
            ZList = BadList
            time.sleep(5)
        ZList = NewList
    else:
        NewList = []
        for Z in ZList:
            TC("Deploy {}".format(Z.Cfg))
            for _ in range(Retries + 1):
                if Z.UpdateStart(Retries=StartRetries) and Z.UpdateWait(Timeout=Timeout):
                    NewList.append(Z)
                    break
                time.sleep(5)
            TC.Close()
        ZList = NewList
    return len(ZList) == len(CfgList)


# ==================================================================================================================


def GetPlatformTag(Str):
    Str = Str.upper()
    if "TRUSTY" in Str or "14.04" in Str:
        return "LINUX_TRUSTY"
    if "XENIAL" in Str or "16.04" in Str:
        return "LINUX_XENIAL"
    if "BIONIC" in Str or "18.04" in Str:
        return "LINUX_BIONIC"
    if "FOCAL" in Str or "20.04" in Str:
        return "LINUX_FOCAL"
    return "LINUX"


class TDebPackage:
    def __init__(self, Token):
        self.SBClient = rest.Client(auth=OAuth(Token))
        self.PackageNum = 0
        self.Task = None
        self.RefURL = None

    def Create(self,
               Branch,
               Revision,
               PkgPaths,
               KeyUser,
               Changelog="",
               Strip=True,
               FullStrip=False,
               PublishTo=[],
               DebianDistribution="unstable",
               DuploadMaxAttempts=3,
               Platform="linux",
               MemoryGb=4,
               DiskGb=50,
               Timeout=3600
               ):
        if not self.SBClient:
            Log("Failed to create sandbox client")
            return 0
        try:
            ArcadiaUrl = "arcadia:/arc/{}/arcadia@{}".format(Branch, Revision)
            Version = "{}.{}".format(Revision, Branch.split("/")[-1])
            Description = "SOLOMON {}".format(" ".join(["{}={}".format(P.split("/")[-2], Version) for P in PkgPaths]))
            Parameters = {
                "checkout": True,
                "checkout_arcadia_from_url": ArcadiaUrl,
                "packages": ";".join(PkgPaths),
                "package_type": "debian",
                "debian_distribution": DebianDistribution,
                "dupload_max_attempts": DuploadMaxAttempts,
                "full_strip_binaries": FullStrip,
                "strip_binaries": Strip,
                "changelog": Changelog,
                "key_user": KeyUser,
                "publish_package": True if len(PublishTo) == 1 else False,
                "publish_to": PublishTo[0] if len(PublishTo) == 1 else "",
                "multiple_publish": True if len(PublishTo) > 1 else False,
                "multiple_publish_to": ";".join(PublishTo) if len(PublishTo) > 1 else ""
            }
            self.Task = self.SBClient.task({
                "owner": "SOLOMON",
                "type": "YA_PACKAGE_2",
                "description": Description,
                "custom_fields": [{"name": K, "value": V} for K, V in Parameters.items()],
                "max_restarts": 3,
                "kill_timeout": Timeout,
                "fail_on_any_error": True,
                "priority": {
                    "class": "SERVICE",
                    "subclass": "HIGH"
                },
                "requirements": {
                    "client_tags": GetPlatformTag(Platform),
                    "platform": "linux",
                    "ram": MemoryGb * (1 << 30),
                    "disk_space": DiskGb * (1 << 30)
                },
                "notifications": []
            })
        except Exception as e:
            Log("Task failed to schedule: {}".format(e))
            return 0
        self.PackageNum = len(PkgPaths)
        self.RefURL = "https://sandbox.yandex-team.ru/task/{}/view".format(self.Task["id"])
        Log("Task id={} status={} scheduled to run ({})".format(self.Task["id"], self.Task["status"], self.RefURL))
        return self.Task["id"]

    def Start(self):
        if self.SBClient and isinstance(self.Task, dict) and self.Task["id"] > 0:
            try:
                Result = self.SBClient.batch.tasks.start.update([self.Task["id"]])
                if Result[0]["status"] == "SUCCESS":
                    Log("Task started: {}".format(Result[0]["message"]))
                    return True
                Log("Bad start result: {}".format(Result))
            except Exception as e:
                Log("Exception during task start: {}".format(e))
        else:
            Log("Nothing to start")
        return False

    def Wait(self):
        if self.SBClient and isinstance(self.Task, dict) and self.Task["id"] > 0:
            while True:
                try:
                    Result = self.SBClient.task[{
                        "limit": 1,
                        "id": self.Task["id"],
                        "fields": "status"
                    }]["items"]
                    if Result[0]["status"] in [
                        "EXCEPTION",
                        "SUCCESS",
                        "EXPIRED",
                        "FAILURE",
                        "TIMEOUT",
                        "NO_RES"
                    ]:
                        Log("Task is done: {} ({})".format(Result[0]["status"], self.RefURL))
                        return Result[0]["status"] in ["SUCCESS"]
                    else:
                        Log("Get status result: {}".format(Result[0]["status"]))
                except Exception as e:
                    Log("Exception during task wait: {}".format(e))
                Log("Sleeping 10 seconds...")
                time.sleep(10)
        else:
            Log("Nothing to wait")
        return False

    def GetPkgs(self):
        if self.SBClient and isinstance(self.Task, dict) and self.Task["id"] > 0 and self.PackageNum > 0:
            Resources = self.SBClient.resource.read(task_id=self.Task["id"], type="YA_PACKAGE", state="READY", limit=self.PackageNum)
            Attrs = (R["attributes"] for R in Resources.get("items", []) if "attributes" in R)
            RealAttrs = (A for A in Attrs if "resource_name" in A and "resource_version" in A)
            return {A["resource_name"]: A["resource_version"] for A in RealAttrs}
        return {}


# ==================================================================================================================


TC = TTC()


def main():
    ConsoleWidth = os.get_terminal_size()[0] if sys.stdout.isatty() else 1024
    Formatter = lambda prog: argparse.RawDescriptionHelpFormatter(prog, width=ConsoleWidth, max_help_position=50)
    Parser = argparse.ArgumentParser(formatter_class=Formatter, description="""
    Start and wait for Sandbox YA_PACKAGE_2 task
    OAuth token could be supplied via env["SANDBOX_TOKEN"]
    Z2 API key could be supplied via env["Z2_API_KEY"]
    Z2 YAV token could be supplied via env["YAV_TOKEN"]
    """)
    Parser.add_argument("--tctag", action="store_true", help="enable teamcity tags")

    Parser.add_argument("--build", action="store_true", help="do sandbox build step")
    Parser.add_argument("--branch", type=str, default=None, metavar="BRANCH", help="branch name or trunk")
    Parser.add_argument("--revision", type=str, default="latest", metavar="REV", help="svn revision (default=%(default)s)")
    Parser.add_argument("--pkg-paths", type=str, default=None, metavar="PKG", help="paths to pkg.json files", nargs="+")
    Parser.add_argument("--user", type=str, default=DefaultKeyRobot, metavar="USER", help="user to sign packages (default=%(default)s)")
    Parser.add_argument("--token", type=str, default=[], metavar="TOKEN", help="sandbox token")
    Parser.add_argument("--changelog", type=str, default="", metavar="TEXT", help="changelog")
    Parser.add_argument("--strip", action="store_true", help="strip binaries")
    Parser.add_argument("--full-strip", action="store_true", help="full strip binaries")
    Parser.add_argument("--publish-to", type=str, default=[], metavar="REPO", help="repo to publish packages (by default do not publish)", nargs="+")
    Parser.add_argument("--distribution", type=str, default="unstable", metavar="DIST", help="distribution to publish packages (default=%(default)s)")
    Parser.add_argument("--dupload-attempts", type=int, default=3, metavar="NUM", help="max duploads attempts (default=%(default)d)")
    Parser.add_argument("--platform", type=str, default="linux", metavar="PLATFORM", help="platform (default=%(default)s)")
    Parser.add_argument("--memory", type=int, default=4, metavar="GB", help="memory to request (default=%(default)d Gb)")
    Parser.add_argument("--disk", type=int, default=50, metavar="GB", help="disk to request (default=%(default)d Gb)")
    Parser.add_argument("--timeout", type=int, default=3600, metavar="SEC", help="task timeout (default=%(default)d sec)")

    Parser.add_argument("--packages", type=str, default=[], metavar="PKG", help="packages to set in Z2 configs", nargs="+")
    Parser.add_argument("--z2-configs", type=str, default=[], metavar="CFG", help="Z2 configs to update (by default do not update)", nargs="+")
    Parser.add_argument("--deploy", action="store_true", help="deploy Z2 configs")
    Parser.add_argument("--parallel", action="store_true", help="deploy Z2 configs in parallel")
    Parser.add_argument("--z2-yav-token", type=str, default=None, metavar="TOKEN", help="yav token")
    Parser.add_argument("--z2-yav-secret-id", type=str, default=None, metavar="ID", help="yav id for secret with Z2 API keys")
    Parser.add_argument("--z2-api-key", type=str, default=None, metavar="KEY", help="Z2 API key")
    Parser.add_argument("--z2-endpoint", type=str, default=None, metavar="HOST", help="Z2 API endpoint")
    Parser.add_argument("--z2-api-retries", type=int, default=4, metavar="NUM", help="Z2 API retries (default=%(default)d)")
    Parser.add_argument("--z2-timeout", type=int, default=900, metavar="NUM", help="Z2 deploy timeout (default=%(default)d sec)")

    Args = Parser.parse_args()

    if Args.tctag:
        TC.Enable()

    Build = Args.build
    Token = Args.token if Args.token else os.environ.get("SANDBOX_TOKEN")
    Branch, Revision = GetBranchLatestRevision(Args.branch)
    Revision = Revision if Args.revision == "latest" else Args.revision.strip("r")
    PkgPaths = Args.pkg_paths
    KeyUser = Args.user
    Changelog = Args.changelog
    Strip = Args.strip
    FullStrip = Args.full_strip
    PublishTo = Args.publish_to
    DebianDistribution = Args.distribution
    DuploadMaxAttempts = Args.dupload_attempts
    Platform = Args.platform
    MemoryGb = Args.memory
    DiskGb = Args.disk
    Timeout = Args.timeout

    Z2Packages = Args.packages
    Z2CfgList = Args.z2_configs
    Z2Deploy = Args.deploy
    Z2Parallel = Args.parallel
    Z2YavToken = Args.token if Args.token else os.environ.get("YAV_TOKEN")
    Z2YavSecretId = Args.z2_yav_secret_id
    Z2ApiKey = Args.z2_api_key if Args.z2_api_key else os.environ.get("Z2_API_KEY")
    Z2Endpoint = Args.z2_endpoint
    Z2Retries = Args.z2_api_retries
    Z2Timeout = Args.z2_timeout

    if Build:
        if not Token:
            try:
                Token = oauth.get_token(ClientID, ClientPassword)
            except Exception as e:
                Log("Failed to get sandbox token: {}".format(e))
                exit(1)
        if not Branch or not Revision or not PkgPaths:
            Log("Cannot start build if no branch, revision or package paths defined")
            exit(1)
    if len(Z2CfgList) > 0 and not Z2YavToken and not Z2YavSecretId and not Z2ApiKey:
        Log("Need to update Z2 configs, but no yav or api keys supplied")
        exit(1)
    Log("Starting task {}".format(json.dumps({
        "Build": Build,
        "Branch": Branch,
        "Revision": Revision,
        "PkgPaths": PkgPaths,
        "KeyUser": KeyUser,
        "Changelog": Changelog,
        "Strip": Strip,
        "FullStrip": FullStrip,
        "PublishTo": PublishTo,
        "DebianDistribution": DebianDistribution,
        "DuploadMaxAttempts": DuploadMaxAttempts,
        "Platform": Platform,
        "MemoryGb": MemoryGb,
        "DiskGb": DiskGb,
        "Timeout": Timeout,
        "Z2Packages": Z2Packages,
        "Z2CfgList": Z2CfgList,
        "Z2Deploy": Z2Deploy,
        "Z2Parallel": Z2Parallel,
        "Z2YavToken": "<present>" if Z2YavToken else "<empty>",
        "Z2YavSecretId": Z2YavSecretId,
        "Z2ApiKey": "<present>" if Z2ApiKey else "<empty>",
        "Z2Endpoint": Z2Endpoint,
        "Z2Retries": Z2Retries,
        "Z2Timeout": Z2Timeout,
    })))

    PkgDict = {}
    try:
        PkgDict = dict(P.split("=") for P in Z2Packages if "=" in P)
    except ValueError:
        Log("Bad package in package list (should be in format package=version): {}".format(Z2Packages))
        exit(1)

    if Build:
        TC("Building packages")
        T = TDebPackage(Token)
        Id = T.Create(
            Branch,
            Revision,
            PkgPaths,
            KeyUser,
            Changelog=Changelog,
            Strip=Strip,
            FullStrip=FullStrip,
            PublishTo=PublishTo,
            DebianDistribution=DebianDistribution,
            DuploadMaxAttempts=DuploadMaxAttempts,
            Platform=Platform,
            MemoryGb=MemoryGb,
            DiskGb=DiskGb,
            Timeout=Timeout
        )
        if Id == 0:
            TC.Close()
            exit(1)
        if not T.Start():
            TC.Close()
            exit(1)
        time.sleep(15)
        if not T.Wait():
            TC.Close()
            exit(1)
        PkgDict.update(T.GetPkgs())
        TC.Close()

    if len(Z2CfgList) > 0:
        if len(PkgDict) == 0:
            Log("Need to deploy, but no packages specified")
            exit(1)
        if not Z2ApiKey and not Z2YavToken:
            try:
                Z2YavToken = oauth.get_token(ClientID, ClientPassword)
            except Exception as e:
                Log("Failed to get yav token: {}".format(e))
                exit(1)
        R = Deploy(
            Z2CfgList,
            PkgDict,
            PushDeploy=Z2Deploy,
            Parallel=Z2Parallel,
            YavToken=Z2YavToken,
            YavSecretId=Z2YavSecretId,
            ApiKey=Z2ApiKey,
            Endpoint=Z2Endpoint,
            Retries=Z2Retries,
            Timeout=Z2Timeout
        )
        if not R:
            exit(1)


if __name__ == "__main__":
    main()
