package main

import (
	"a.yandex-team.ru/solomon/libs/go/conductor"
	"bufio"
	"context"
	"fmt"
	"go.uber.org/atomic"
	"log"
	"net"
	"os"
	"strings"
	"time"
)

const (
	TAG = "yasm_monitored"
)

type HostInfo struct {
	Host     string
	Groups   []string
	HasAgent bool
	Err      error
}

func (h *HostInfo) WriteTo(w *bufio.Writer) error {
	if h.HasAgent {
		if _, err := w.WriteString("PRESENT "); err != nil {
			return err
		}
	} else {
		if _, err := w.WriteString("ABSENT "); err != nil {
			return err
		}
	}

	if _, err := w.WriteString(h.Host); err != nil {
		return err
	}

	if err := w.WriteByte(' '); err != nil {
		return err
	}

	for i, group := range h.Groups {
		if i > 0 {
			if err := w.WriteByte(','); err != nil {
				return err
			}
		}
		if _, err := w.WriteString(group); err != nil {
			return err
		}
	}

	return nil
}

type GroupInfo struct {
	Name     string
	Projects []string
	HasAgent map[string]bool
}

func NewGroupInfo(name string) *GroupInfo {
	return &GroupInfo{Name: name, HasAgent: make(map[string]bool)}
}

func (g *GroupInfo) AnyHasAgent() bool {
	for _, hasAgent := range g.HasAgent {
		if hasAgent {
			return true
		}
	}
	return false
}

func (g *GroupInfo) SameHosts(hosts []string) bool {
	if len(hosts) != len(g.HasAgent) {
		return false
	}

	for _, host := range hosts {
		if _, present := g.HasAgent[host]; !present {
			return false
		}
	}

	return true
}

func loadHosts(conClient *conductor.Client) (<-chan string, int) {
	log.Println("resolving conductor tag", TAG)

	hosts, err := conClient.TagToHosts(context.Background(), TAG)
	if err != nil {
		log.Fatalf("cannot resolve %s tag: %v", TAG, err)
	}

	hostsCh := make(chan string, 10)
	go func() {
		for _, host := range hosts {
			hostsCh <- host
		}
		close(hostsCh)
	}()

	return hostsCh, len(hosts)
}

func resolveGroups(parallelism int32, conClient *conductor.Client, in <-chan string) <-chan HostInfo {
	out := make(chan HostInfo, parallelism)
	activeWorkers := atomic.NewInt32(parallelism)

	for i := int32(0); i < parallelism; i++ {
		go func() {
			ctx := context.Background()
			for host := range in {
				groups, err := conClient.HostToGroups(ctx, host)
				if err != nil {
					out <- HostInfo{Host: host, Err: err}
					break
				} else {
					out <- HostInfo{Host: host, Groups: groups}
				}
			}

			if activeWorkers.Dec() == 0 {
				// last active worker will close output channel
				close(out)
			}
		}()
	}

	return out
}

func checkAgent(parallelism int32, in <-chan HostInfo) <-chan HostInfo {
	out := make(chan HostInfo, parallelism)
	activeWorkers := atomic.NewInt32(parallelism)

	for i := int32(0); i < parallelism; i++ {
		go func() {
			for h := range in {
				if h.Err != nil {
					// propagate error
					out <- h
					break
				}

				conn, err := net.DialTimeout("tcp", h.Host+":11003", time.Second*3)
				if err == nil {
					_ = conn.Close()
					h.HasAgent = true
				} else {
					h.HasAgent = false
				}
				out <- h
			}

			if activeWorkers.Dec() == 0 {
				// last active worker will close output channel
				close(out)
			}
		}()
	}

	return out
}

func loadFromNetwork(conClient *conductor.Client, filename string) (map[string]*GroupInfo, error) {
	hostsCh, hostsCount := loadHosts(conClient)
	withGroupsCh := resolveGroups(10, conClient, hostsCh)
	withAgentsCh := checkAgent(50, withGroupsCh)

	log.Println("will write result into", filename)
	f, err := os.Create(filename)
	if err != nil {
		log.Fatalf("cannot create file %s: %v", filename, err)
	}
	w := bufio.NewWriter(f)

	defer func() {
		_ = w.Flush()
		_ = f.Close()
	}()

	groups := make(map[string]*GroupInfo, 1000)

	done := 0
	log.Printf("checking all %d hosts", hostsCount)
	for h := range withAgentsCh {
		if h.Err != nil {
			return nil, fmt.Errorf("unexpected error while processing host %s: %v", h.Host, h.Err)
		}

		if err = h.WriteTo(w); err != nil {
			return nil, fmt.Errorf("cannot write %s data to file %s", h.Host, filename)
		}
		if err = w.WriteByte('\n'); err != nil {
			return nil, fmt.Errorf("cannot write %s data to file %s", h.Host, filename)
		}

		for _, group := range h.Groups {
			info := groups[group]
			if info == nil {
				info = NewGroupInfo(group)
				groups[group] = info
			}
			info.HasAgent[h.Host] = h.HasAgent
		}

		done++
		_, _ = fmt.Fprintf(os.Stderr, "\033[1000DDone %d / Left %d", done, hostsCount-done)
	}

	_, _ = fmt.Fprint(os.Stderr, "\033[1000D")
	log.Println("OK")

	return groups, nil
}

func loadFromFile(filename string) (map[string]*GroupInfo, error) {
	log.Println("loading host information from", filename)
	f, err := os.Open(filename)
	if err != nil {
		return nil, fmt.Errorf("cannot open file %s: %w", filename, err)
	}
	defer f.Close()

	groupsMap := make(map[string]*GroupInfo, 1000)

	count := 0
	s := bufio.NewScanner(f)
	for s.Scan() {
		line := strings.Split(s.Text(), " ")
		hasAgent := line[0] == "PRESENT"
		host := line[1]
		groups := strings.Split(line[2], ",")

		for _, group := range groups {
			info := groupsMap[group]
			if info == nil {
				info = NewGroupInfo(group)
				groupsMap[group] = info
			}
			info.HasAgent[host] = hasAgent
		}
		count++
	}

	if err = s.Err(); err != nil {
		return nil, fmt.Errorf("cannot read data from %s: %w", filename, err)
	}

	log.Printf("loaded %d hosts", count)

	return groupsMap, nil
}

func filterGroups(parallelism int32, conClient *conductor.Client, groups <-chan *GroupInfo) <-chan *GroupInfo {
	out := make(chan *GroupInfo, parallelism)
	activeWorkers := atomic.NewInt32(parallelism)

	for i := int32(0); i < parallelism; i++ {
		go func() {
			for group := range groups {
				if group.AnyHasAgent() {
					continue
				}

				hosts, err := conClient.GroupToHosts(context.Background(), group.Name)
				if err != nil {
					log.Fatalf("cannot resolve conductor group %s: %v", group.Name, err)
				}

				if group.SameHosts(hosts) {
					group.Projects, err = conClient.GroupToProjects(context.Background(), group.Name)
					if err != nil {
						log.Fatalf("cannot get projects for conductor group %s: %v", group.Name, err)
					}

					out <- group
				}
			}

			if activeWorkers.Dec() == 0 {
				// last active worker will close output channel
				close(out)
			}
		}()
	}

	return out
}

func main() {
	if len(os.Args) != 2 {
		fmt.Printf("Usage: %s <filename>\n", os.Args[0])
		os.Exit(1)
	}

	filename := os.Args[1]
	conClient := conductor.NewClientWithTimeout(2 * time.Minute)

	var groups map[string]*GroupInfo
	var err error

	if _, err = os.Stat(filename); err != nil && os.IsNotExist(err) {
		groups, err = loadFromNetwork(conClient, filename)
	} else {
		groups, err = loadFromFile(filename)
	}

	if err != nil {
		log.Fatal(err)
	}

	log.Printf("loaded %d groups", len(groups))

	groupsCh := make(chan *GroupInfo, 10)
	go func() {
		for _, group := range groups {
			groupsCh <- group
		}
		close(groupsCh)
	}()

	log.Println("looking for groups without agent on all hosts")

	count := 0
	badGroupsCh := filterGroups(10, conClient, groupsCh)
	for group := range badGroupsCh {
		fmt.Println(strings.Join(group.Projects, ","), group.Name)
		count++
		_, _ = fmt.Fprintf(os.Stderr, "\033[1000DFound %d such groups", count)
	}
	_, _ = fmt.Fprintf(os.Stderr, "\033[1000DFound %d such groups\n", count)
}
