/* eslint-disable key-spacing,max-len,guard-for-in,no-restricted-syntax */
import { computeRangeFromParamsForOldUi } from '../../pages/data/utils';
import { getSelectorsAsEntriesFromParams } from '../../utils/queryArgs';
import { formatSearch } from '../../utils/url';
import { isTrue } from '../../utils/boolean';
import * as InternalAPI from '../internal';
import { createGggSelectors } from './gggSelectors';
import { fetchDashboard } from '../dashboards';
import StringInterpolator from '../../utils/StringInterpolator';
import UserLinksBasic from '../../utils/UserLinksBasic';
import createGggNavButtons from './createGggNavButtons';
import createGggTimeRange from './createGggTimeRange';
import createMatchingParamsFromSelectorsAsEntries
  from './createMatchingParamsFromSelectorsAsEntries';

function transformToOldItem(panel, parametersMap) {
  let title = StringInterpolator.interpolatePattern(panel.title || '', parametersMap, true).trim();
  const subtitle = StringInterpolator.interpolatePattern(panel.subtitle || '', parametersMap, true).trim();
  const urlSubstituted = StringInterpolator.interpolatePattern(panel.url || '', parametersMap, true).trim();
  const markdown = StringInterpolator.interpolatePattern(panel.markdown || '', parametersMap, true).trim();

  const solomonUrl = UserLinksBasic.fixSolomonUrl(urlSubstituted);

  if (!title && UserLinksBasic.isSolomonUrl(panel.url)) {
    title = '???';
  }

  const url = UserLinksBasic.enforceGraphOnly(solomonUrl);
  const rawUrl = UserLinksBasic.removeGraphOnly(solomonUrl);

  return {
    type: panel.type,
    title,
    subtitle,
    url,
    rawUrl,
    rowspan: panel.rowspan || 0,
    colspan: panel.colspan || 0,
    markdown,
  };
}

function transformToOldRow(row, parametersMap) {
  const panels = row.panels || [];
  const items = panels.map((panel) => transformToOldItem(panel, parametersMap));
  return { items };
}

function transformToOldDashboard(dashboard, parametersMap) {
  const title = StringInterpolator.interpolatePattern(dashboard.name || '', parametersMap, true);
  const rows = dashboard.rows || [];
  const oldRows = rows.map((row) => transformToOldRow(row, parametersMap));
  return {
    title,
    rows: oldRows,
    height: dashboard.heightMultiplier || 1,
    description: dashboard.description || '',
    generatedId: dashboard.generatedId,
  };
}

async function loadDashboardData(project, dashboardId, params, selectorsAsEntries, matchingParams) {
  const dashboard = await fetchDashboard(project, dashboardId);

  const projectParameters = dashboard.parameters.filter((entry) => entry.name === 'project');
  const dataProject = projectParameters.length > 0 ? projectParameters[0].value : project;

  const parametersMap = {};
  dashboard.parameters.forEach((entry) => {
    parametersMap[entry.name] = params[entry.name] || params[`l.${entry.name}`] || '';
  });
  parametersMap.project = dataProject;

  const matchingDashboards = await InternalAPI.getMatchingDashboards(project, matchingParams);

  const entityOpts = {
    name: dashboard.name,
    matchingEntities: matchingDashboards,
  };

  const oldSelectorsData = createGggSelectors(project, dataProject, selectorsAsEntries, 'dashboard', dashboardId, entityOpts);

  const oldDashboard = transformToOldDashboard(dashboard, parametersMap);

  const graphTitle = oldDashboard.title || dashboard.name || dashboardId || 'Dashboard';
  const windowTitle = graphTitle;

  return {
    windowTitle,
    graphTitle,
    dashboard: oldDashboard,
    oldSelectorsData,
  };
}

export async function loadDashboard(params, reload = false) {
  const project = params.project || '';
  const b = params.b || '1d';
  const e = params.e || '';
  const dashboardId = params.dashboard || '';

  const nowMillis = Date.now();

  const url = `/?${formatSearch(params)}`;

  const { fromMillis, toMillis } = computeRangeFromParamsForOldUi(b, e, nowMillis);

  const graphOnly = isTrue(params.graphOnly);
  const loadFirstPage = !graphOnly && !reload;

  const selectorsAsEntries = getSelectorsAsEntriesFromParams(params);

  const matchingParams = createMatchingParamsFromSelectorsAsEntries(project, selectorsAsEntries);

  let matchingMenus;
  let currentPageIdInHistory = '';

  if (loadFirstPage) {
    matchingMenus = await InternalAPI.getMatchingMenus(project, matchingParams);

    const pinResult = await InternalAPI.getPinByUrl({ url });
    currentPageIdInHistory = pinResult ? pinResult.currentPageIdInHistory : '';
  } else {
    matchingMenus = [];
  }

  const genericResp = await loadDashboardData(project, dashboardId, params, selectorsAsEntries, matchingParams);

  const {
    windowTitle,
    graphTitle,
    dashboard,
    oldSelectorsData,
  } = genericResp;

  const timeRange = createGggTimeRange(fromMillis, toMillis, nowMillis);

  const navButtons = createGggNavButtons(b, e, fromMillis, toMillis);

  return {
    windowTitle,
    graphTitle,
    selectors: oldSelectorsData,
    menu: matchingMenus,
    navButtons,
    dashboard,
    fullItem: null,
    timeRange,
    currentPageIdInHistory,
  };
}

export default {};
