/* eslint-disable key-spacing,max-len,guard-for-in,no-restricted-syntax */
import flatMap from 'lodash/flatMap';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';
import { computeRangeFromParamsForOldUi } from '../../pages/data/utils';
import { getSelectorsAsEntriesFromParams } from '../../utils/queryArgs';
import { formatSearch } from '../../utils/url';
import * as DataAPI from '../data';
import * as GraphAPI from '../graphs';

import { fixSeries, LINE_NAMING_MODE } from '../../utils/data';
import { formatOldSelectorEntriesAsNewSelectors } from '../../utils/SelectorUtils';
import GraphSettings from '../../pages/old/OldGraphPage/utils/GraphSettings';
import GraphDataWithStackConfigAndCookie
  from '../../solomon-graph/stack/GraphDataWithStackConfigAndCookie';
import StackBuilder from '../../solomon-graph/stack/StackBuilder';
import Interpolate from '../../solomon-graph/stack/Interpolate';
import GraphConfUtils from '../../pages/old/OldGraphPage/utils/GraphConfUtils';
import parseBoolean, { isTrue } from '../../utils/boolean';
import SensorsChecksSet from '../../pages/old/OldGraphPage/OldLegend/SensorsChecksSet';
import ColorSchemeType from '../../utils/color/ColorSchemeType';
import { parseDuration } from '../../utils/duration';
import UserLinksBasic from '../../utils/UserLinksBasic';
import * as InternalAPI from '../internal';
import { createGggSelectors } from './gggSelectors';
import createGggNavButtons from './createGggNavButtons';
import createGggTimeRange from './createGggTimeRange';
import createMatchingParamsFromSelectorsAsEntries
  from './createMatchingParamsFromSelectorsAsEntries';
import {
  computeInterpolationForAutoGraph,
  computeStackForAutoGraph,
  createElementsWithExpressionList,
  transformProgram,
} from '../../pages/old/utils/graphToExpressionTransformations';
import StringInterpolator from '../../utils/StringInterpolator';
import GraphConverter from '../../pages/old/OldGraphPage/utils/GraphConverter';
import PROJECT_CONFIGS from '../../pages/old/utils/projectConfigs';

const LEFT_YAXIS = { name: 'left', positionValue: 'left' };
const RIGHT_YAXIS = { name: 'right', positionValue: 'right' };

function createMetaData(series, elements, globalStack, parsedChecks) {
  const sensorLabels = series.rawSeries.labels || {};

  let link = '';
  if (!isEmpty(sensorLabels)) {
    const search = new URLSearchParams('');

    entries(sensorLabels).forEach((entry) => {
      let key = entry[0];
      const value = entry[1];

      if (key !== 'project' && key !== 'cluster' && key !== 'service') {
        key = `l.${key}`;
      }

      search.append(key, value);
    });

    search.append(UserLinksBasic.GRAPH_PARAM, UserLinksBasic.GRAPH_VALUE_AUTO);

    link = `/?${search.toString()}`;
  }

  const { refIndex } = series;
  const element = elements[refIndex] || null;

  let finalElementStackName;
  let finalElementArea;
  let down;
  let color;
  let yaxisConf;

  if (element) {
    if (element.area !== undefined && element.area !== null) {
      finalElementArea = element.area;
    } else {
      finalElementArea = globalStack;
    }
    finalElementStackName = element.stack || (finalElementArea ? 'stack0' : '');
    down = element.down || false;
    if (element.color) {
      color = element.color;
    } else {
      color = series.color;
    }
    if (element.yaxis === 'RIGHT') {
      yaxisConf = RIGHT_YAXIS;
    } else {
      yaxisConf = LEFT_YAXIS;
    }
  } else {
    finalElementArea = globalStack;
    finalElementStackName = finalElementArea ? 'stack0' : '';
    down = false;
    color = series.color;
    yaxisConf = LEFT_YAXIS;
  }

  const show = parsedChecks.include(series.name);

  return ({
    salmonId: series.name,
    label: series.name,
    show,
    color,
    legendNames: series.legendNames,
    area: finalElementArea,
    stack: finalElementStackName,
    down,
    link,
    sensorLabels,
    salmonDebug: {},
    yaxisConf,
  });
}

export function createGraphData(fixedSeries, elements, globalStack, interpolate, norm, aggr, parsedChecks) {
  const rawSeries = fixedSeries
    .map((series) => {
      const metaData = createMetaData(series, elements, globalStack, parsedChecks);

      const { show } = metaData;

      let rawData;
      let salmonStats;

      if (show) {
        if (series.rawSeries.__oldStats) {
          rawData = [];
          const oldStats = series.rawSeries.__oldStats;
          salmonStats = {
            max: oldStats.max,
            min: oldStats.min,
            sum: oldStats.sum,
            avg: oldStats.avg,
            last: oldStats.last,
            hasData: true,
          };
        } else {
          rawData = [];
          for (let i = 0; i < series.values.length; ++i) {
            const ts = Math.trunc(series.timestamps[i] / 1000);
            const value = series.values[i];
            rawData.push([ts, value]);
          }

          const pointValues = series.values.filter((x) => !isNaN(x) && Number.isFinite(x));

          const max = pointValues.length === 0
            ? NaN
            : pointValues.reduce((a1, a2) => Math.max(a1, a2));
          const min = pointValues.length === 0
            ? NaN
            : pointValues.reduce((a1, a2) => Math.min(a1, a2));
          const sum = pointValues.length === 0 ? NaN : pointValues.reduce((a1, a2) => a1 + a2);
          const avg = pointValues.length === 0 ? NaN : sum / pointValues.length;
          const last = pointValues.length === 0 ? NaN : pointValues[pointValues.length - 1];

          salmonStats = {
            max, min, sum, avg, last, hasData: true,
          };
        }
      } else {
        rawData = [];
        salmonStats = {
          max: NaN, min: NaN, sum: NaN, avg: NaN, last: NaN, hasData: false,
        };
      }

      let aggrValue;
      switch (aggr) {
        case 'max':
          aggrValue = salmonStats.max;
          break;
        case 'min':
          aggrValue = salmonStats.min;
          break;
        case 'sum':
          aggrValue = salmonStats.sum;
          break;
        case 'avg':
          aggrValue = salmonStats.avg;
          break;
        case 'last':
          aggrValue = salmonStats.last;
          break;
        default:
          aggrValue = NaN;
      }

      return ({
        ...metaData,
        aggregatedValue: aggrValue,
        rawData,
        data: rawData,
        salmonStats,
      });
    });

  const inputs = rawSeries
    .filter((s) => s.show)
    .map((s) => {
      const gd = {
        timestamps: s.rawData.map((p) => p[0] * 1000),
        values: s.rawData.map((p) => p[1]),
      };
      return new GraphDataWithStackConfigAndCookie(
        gd,
        s.stack,
        s.down,
        s,
        s.yaxisConf.positionValue,
      );
    });

  try {
    const stacks = StackBuilder.makeStacks(inputs, interpolate, norm);

    const stackedLineById = {};

    stacks.forEach((l) => { stackedLineById[l[1].salmonId] = l; });

    return rawSeries.map((s) => {
      const stackedLine = stackedLineById[s.salmonId];

      if (stackedLine) {
        const stripe = stackedLine[0];
        const cookie = stackedLine[1];

        let data;
        if (cookie.area) {
          data = stripe.getPointsHighLow();
        } else {
          data = stripe.getPointsHigh();
        }

        return ({
          ...cookie,
          data,
        });
      }

      return s;
    });
  } catch (e) {
    console.error(e);
    return rawSeries;
  }
}

export function computeDownsamplingParams(config, points) {
  const downsampling = {};
  const downsamplingType = GraphSettings.downsampling.getFromOrDefaultStrictly(config);
  const downsamplingAggr = GraphSettings.downSamplingAggr.getFromOrDefaultStrictly(config);
  const downsamplingFill = GraphSettings.downsamplingFill.getFromOrDefaultStrictly(config);
  const ignoreMinStepMillis = GraphSettings.ignoreMinStepMillis.getBoolFrom(config);

  if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_OFF) {
    downsampling.disabled = true;
    return downsampling;
  }
  if (downsamplingAggr && downsamplingAggr !== 'default') {
    downsampling.aggregation = downsamplingAggr.toUpperCase();
  }
  if (downsamplingFill && downsamplingFill !== 'null') {
    downsampling.fill = downsamplingFill.toUpperCase();
  }
  if (ignoreMinStepMillis) {
    downsampling.ignoreMinStepMillis = true;
  }
  if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_BY_POINTS) {
    downsampling.maxPoints = parseInt(config.maxPoints, 10) || 1000;
  } else if (downsamplingType === UserLinksBasic.DOWNSAMPLING_PARAM_BY_INTERVAL) {
    downsampling.gridMillis = parseDuration(config.grid || '1h');
  } else {
    downsampling.maxPoints = points;
  }
  return downsampling;
}

function isTimeseriesVector(data) {
  return data.vector && data.vector.every((e) => !!e.timeseries);
}

async function loadAutoGraphData(project, params, fromMillis, toMillis, forceCluster) {
  const selectorsList = getSelectorsAsEntriesFromParams(params);

  const wrappedSelectors = formatOldSelectorEntriesAsNewSelectors(selectorsList, true);

  const program = transformProgram(params, wrappedSelectors);

  const downsampling = computeDownsamplingParams(params, params.points);

  const dataParams = {
    program,
    from: fromMillis,
    to: toMillis,
    downsampling,
    __oldMode: true,
  };

  if (forceCluster) {
    dataParams.forceCluster = forceCluster;
  }

  if (params.checks) {
    dataParams.__checks = params.checks;
  }

  const data = await DataAPI.fetchSensorsData(project, dataParams);

  let timeseriesList;
  if (data.timeseries) {
    timeseriesList = [{ ...data.timeseries, refIndex: 0 }];
  } else if (isTimeseriesVector(data)) {
    timeseriesList = data.vector.map((e) => ({ ...e.timeseries, refIndex: 0 }));
  } else {
    timeseriesList = [];
  }

  return {
    timeseriesList,
    elements: [null],
    truncated: data.__truncated || false,
    summary: data.__summary || false,
  };
}

async function loadAutoGraphGenericData(project, params, selectorsAsEntries) {
  const cluster = params.cluster || '*';
  const service = params.service || '*';

  const oldSelectorsData = createGggSelectors(project, project, selectorsAsEntries, 'graph', UserLinksBasic.GRAPH_VALUE_AUTO);

  const projectGraphConf = PROJECT_CONFIGS[project] || {};
  const graphConfFromParams = GraphConfUtils.fromQueryArgParams(params);
  const graphConf = GraphConfUtils.merge(projectGraphConf, graphConfFromParams);

  const windowTitle = `${project}_${cluster}_${service}`;
  const graphTitle = 'Graph';

  return {
    dataProjectId: project,
    windowTitle,
    graphTitle,
    graphConf,
    oldSelectorsData,
  };
}

async function loadConfiguredGenericData(project, graphId, params, selectorsAsEntries, matchingParams, loadFirstPage) {
  const graph = await GraphAPI.fetchGraph(project, graphId);

  const projectParameters = graph.parameters.filter((entry) => entry.name === 'project');
  const dataProject = projectParameters.length > 0 ? projectParameters[0].value : project;

  let matchingGraphs;

  if (loadFirstPage) {
    matchingGraphs = await InternalAPI.getMatchingGraphs(project, matchingParams);
  } else {
    matchingGraphs = [];
  }

  const graphName = graph.name || graphId || 'Graph';

  const entityOpts = {
    name: graph.name,
    matchingEntities: matchingGraphs,
  };

  const oldSelectorsData = createGggSelectors(project, dataProject, selectorsAsEntries, 'graph', graphId, entityOpts);

  const projectGraphConf = PROJECT_CONFIGS[project] || {};
  const graphConfFromGraph = GraphConverter.fromModel(graph);
  const graphConfFromParams = GraphConfUtils.fromQueryArgParams(params);

  delete graphConfFromParams.stack;

  let graphConf = GraphConfUtils.merge(projectGraphConf, graphConfFromGraph);
  graphConf = GraphConfUtils.merge(graphConf, graphConfFromParams);

  const graphTitle = graphName;
  const windowTitle = graphName;

  return {
    dataProjectId: dataProject,
    windowTitle,
    graphTitle,
    graphConf,
    oldSelectorsData,
    graph,
  };
}

function createAliasForTimeseries(timeseries, elementTitle, labels) {
  let newAlias;
  if (timeseries.alias) {
    newAlias = timeseries.alias;
  } else {
    newAlias = elementTitle ? StringInterpolator.interpolatePattern(elementTitle, labels) : '';
  }
  return newAlias;
}

function toDataWithExprList(resp, element, refIndex) {
  if (!resp) {
    return [];
  }

  const elementTitle = (element && element.title) ? element.title : '';

  if (resp.timeseries) {
    const { timeseries } = resp;
    const labels = timeseries.labels || {};
    timeseries.alias = createAliasForTimeseries(timeseries, elementTitle, labels);
    timeseries.refIndex = refIndex;
    return [timeseries];
  }

  if (isTimeseriesVector(resp)) {
    return resp.vector.map((el) => {
      const { timeseries } = el;
      const labels = timeseries.labels || {};
      timeseries.alias = createAliasForTimeseries(timeseries, elementTitle, labels);
      timeseries.refIndex = refIndex;
      return timeseries;
    });
  }

  return [];
}

async function loadConfiguredGraphData(params, project, graph, fromMillis, toMillis, forceCluster) {
  const projectParameters = graph.parameters.filter((entry) => entry.name === 'project');

  const dataProject = projectParameters.length > 0 ? projectParameters[0].value : project;

  const graphConfFromGraph = GraphConverter.fromModel(graph);

  const graphConfFromParams = GraphConfUtils.fromQueryArgParams(params);

  delete graphConfFromParams.stack;

  const graphConf = GraphConfUtils.merge(graphConfFromGraph, graphConfFromParams);

  const parametersMap = {};
  graph.parameters.forEach((entry) => {
    parametersMap[entry.name] = params[entry.name] || params[`l.${entry.name}`] || '';
  });
  parametersMap.project = dataProject;

  const elementsWithExpressionList = createElementsWithExpressionList(dataProject, parametersMap, graph, graphConf);

  const elements = elementsWithExpressionList.map((elementWithExpression) => elementWithExpression.element);

  const downsampling = computeDownsamplingParams(graphConf, params.points);

  const supportOldMode = graph.elements.length === 1 && graph.elements[0].type === 'SELECTORS';

  const dataPromises = elementsWithExpressionList.map((elementWithExpr, refIndex) => {
    const dataParams = {
      program: elementWithExpr.expression,
      from: fromMillis,
      to: toMillis,
      downsampling,
    };
    if (forceCluster) {
      dataParams.forceCluster = forceCluster;
    }
    if (supportOldMode) {
      dataParams.__oldMode = true;
      if (params.checks) {
        dataParams.__checks = params.checks;
      }
    }
    return DataAPI.fetchSensorsData(dataProject, dataParams).then((response) => ({
      response,
      element: elementWithExpr.element,
      refIndex,
    }));
  });

  const listOfDataResponses = await Promise.all(dataPromises);

  let listOfTimeseriesList;
  let truncated = false;
  let summary = false;

  if (listOfDataResponses.length === 0) {
    listOfTimeseriesList = [];
  } else {
    listOfTimeseriesList = listOfDataResponses.map((resp) => toDataWithExprList(resp.response, resp.element, resp.refIndex));

    if (supportOldMode) {
      const singleDataResponse = listOfDataResponses[0].response;
      truncated = singleDataResponse.__truncated || false;
      summary = singleDataResponse.__summary || false;
    }
  }

  const timeseriesList = flatMap(listOfTimeseriesList);

  return {
    timeseriesList,
    elements,
    truncated,
    summary,
  };
}

export async function loadGeneric(params, reload = false) {
  const project = params.project || '';
  const b = params.b || '1d';
  const e = params.e || '';
  const graphParam = params.graph;

  const nowMillis = Date.now();

  const url = `/?${formatSearch(params)}`;

  const { fromMillis, toMillis } = computeRangeFromParamsForOldUi(b, e, nowMillis);

  const graphOnly = isTrue(params.graphOnly);
  const loadFirstPage = !graphOnly && !reload;

  const selectorsAsEntries = getSelectorsAsEntriesFromParams(params);

  const matchingParams = createMatchingParamsFromSelectorsAsEntries(project, selectorsAsEntries);

  let matchingMenus;
  let currentPageIdInHistory = '';

  if (loadFirstPage) {
    matchingMenus = await InternalAPI.getMatchingMenus(project, matchingParams);

    const pinResult = await InternalAPI.getPinByUrl({ url });
    currentPageIdInHistory = pinResult ? pinResult.currentPageIdInHistory : '';
  } else {
    matchingMenus = [];
  }

  const genericResp = graphParam === UserLinksBasic.GRAPH_VALUE_AUTO
    ? await loadAutoGraphGenericData(project, params, selectorsAsEntries)
    : await loadConfiguredGenericData(project, graphParam, params, selectorsAsEntries, matchingParams, loadFirstPage);

  const {
    windowTitle, graphTitle, graphConf, graph, oldSelectorsData, dataProjectId,
  } = genericResp;

  const description = graphConf.description || '';

  const fullItem = {
    // Common
    graphConf,
    description,
    // New in experimental mode
    graph,
    dataProjectId,
  };

  const timeRange = createGggTimeRange(fromMillis, toMillis, nowMillis);

  const navButtons = createGggNavButtons(b, e, fromMillis, toMillis);

  return {
    windowTitle,
    graphTitle,
    selectors: oldSelectorsData,
    menu: matchingMenus,
    navButtons,
    fullItem,
    timeRange,
    currentPageIdInHistory,
  };
}

export async function loadGraphData(params, fullItem, fromMillis, toMillis, nowMillis) {
  const { graphConf, graph } = fullItem;

  const project = params.project || '';
  const graphParam = params.graph;

  const { forceCluster } = params;

  const selectorsAsEntries = getSelectorsAsEntriesFromParams(params);

  const matchingParams = createMatchingParamsFromSelectorsAsEntries(project, selectorsAsEntries);

  const shardHealth = await InternalAPI.getMatchingShard(project, matchingParams);

  const isAutoGraph = graphParam === UserLinksBasic.GRAPH_VALUE_AUTO;

  const dataResponse = isAutoGraph
    ? await loadAutoGraphData(project, params, fromMillis, toMillis, forceCluster)
    : await loadConfiguredGraphData(
      params, project, graph, fromMillis, toMillis, forceCluster,
    );

  const {
    timeseriesList, elements, truncated, summary,
  } = dataResponse;

  const criticalErrors = [];
  let noncriticalError = null;
  let hideGraph = false;

  if (!isAutoGraph && timeseriesList.length > 640 && !summary) {
    hideGraph = true;
    timeseriesList.length = 0;
    criticalErrors.push({
      id: 'too_many_series_in_configured_graph',
      text: 'Too many metrics are loaded, expected: 640',
      alarm: true,
    });
  }

  const globalStack = parseBoolean(computeStackForAutoGraph(project, graphConf));
  const interpolate = Interpolate.parse(computeInterpolationForAutoGraph(project, graphConf));
  const norm = GraphSettings.norm.getBoolFrom(graphConf);
  const aggr = GraphSettings.aggr.getFromOrDefaultStrictly(graphConf);

  const checks = params.checks || '';
  const parsedChecks = SensorsChecksSet.parse(checks);

  const colorSchemeParams = {
    type: GraphSettings.cs.getFromOrDefaultStrictly(graphConf) || ColorSchemeType.AUTO,
    green: GraphSettings.green.getFromOrDefault(graphConf),
    yellow: GraphSettings.yellow.getFromOrDefault(graphConf),
    red: GraphSettings.red.getFromOrDefault(graphConf),
    violet: GraphSettings.violet.getFromOrDefault(graphConf),
  };

  const fixParams = { colorScheme: colorSchemeParams, mode: LINE_NAMING_MODE.OLD_UI };

  const fixedSeries = fixSeries(timeseriesList, fixParams);

  const { lines } = fixedSeries;
  const columns = fixedSeries.columns.map((c) => c.title);

  if (timeseriesList.length === 0 && !hideGraph) {
    noncriticalError = {
      text: 'No lines found',
    };
    hideGraph = true;
  } else if (lines.length === 0 && !hideGraph) {
    // Specially for complex data types like empty histograms
    noncriticalError = {
      text: 'No data in the time interval',
    };
    hideGraph = true;
  }

  if (truncated) {
    criticalErrors.push({
      id: 'truncated_alert',
      text: 'More than 10000 series found, Solomon will only display truncated result in deprecated mode. Please use limit parameter instead of',
      alarm: false,
    });
  }
  if (summary) {
    const graphMode = GraphSettings.graphMode.getFromOrDefault(graphConf);
    const isAggregatedGraph = graphMode && graphMode.toLowerCase() !== 'graph';
    if (!isAggregatedGraph) {
      criticalErrors.push({
        id: 'too_many_lines_alert',
        text: 'More than 640 series found, Solomon will only display aggregated result in deprecated mode. Please use more concrete selectors or reconsider your metrics and their visualization.',
        alarm: false,
      });
    }
  }

  const graphGraphData = createGraphData(lines, elements, globalStack, interpolate, norm, aggr, parsedChecks);

  // Necessary to support default graph conf in old API
  const defaultGraphConf = isAutoGraph ? (PROJECT_CONFIGS[project] || {}) : graphConf;

  return {
    start: new Date(fromMillis).toISOString(),
    end: new Date(toMillis).toISOString(),
    now: new Date(nowMillis).toISOString(),
    graphConf: defaultGraphConf,
    graphData: graphGraphData,
    health: shardHealth,
    // New in experimental mode
    noncriticalError,
    criticalErrors,
    columns,
    hideGraph,
  };
}
