/* eslint-disable prefer-promise-reject-errors */
import 'whatwg-fetch';

import Auth from '../auth/Auth';
import { formatSearch } from '../utils/url';
import FRONTEND_VERSION from '../utils/frontendVersion';
import { saveLocation } from '../utils/sessionStorage';
import { isAsUserAuth } from '../utils/env';

function checkServerVersion(response) {
  const serverVersion = response.headers.get('X-Solomon-Version');
  if (serverVersion !== null
    && serverVersion !== undefined
    && serverVersion !== FRONTEND_VERSION) {
    console.warn(`Reloading page since gateway reported version [${serverVersion}]`
      + ` which is different from frontend version [${FRONTEND_VERSION}]`);
    if (window) {
      window.setInterval(() => {
        if (navigator.onLine) {
          window.location.reload(true);
        } else {
          console.warn('Cannot reload page because browser is offline, try again later');
        }
      }, 60000);
    }
  }
}

function resetSessionId() {
  const options = {
    credentials: 'include',
  };

  return new Promise((resolve, reject) => {
    fetch('https://passport.yandex-team.ru/auth/update', options)
      .then((response) => {
        if (response.status === 200) {
          return response.json();
        }

        return reject();
      })
      .then((json) => {
        if (json.status === 'ok') {
          return resolve();
        }

        return reject();
      })
      .catch(() => reject());
  });
}

function getAuthHeaders() {
  if (isAsUserAuth()) {
    return { Authorization: `AsUser ${Auth.state.login()}` };
  }

  return {};
}

function setHeaders(options, customHeader) {
  const authHeaders = getAuthHeaders();

  if (customHeader) {
    if (options.headers) {
      Object.assign(options.headers, authHeaders);
    } else {
      Object.assign(options, { headers: authHeaders });
    }
  } else {
    Object.assign(options, {
      headers: {
        ...authHeaders,
        Accept: 'application/json',
        'Content-Type': 'application/json',
      },
    });
  }
}

export function sendRequest(url, options = {}, customHeader = false, repeated = false) {
  setHeaders(options, customHeader);
  // eslint-disable-next-line no-param-reassign
  options.credentials = 'same-origin';

  return new Promise((resolve, reject) => {
    fetch(url, options)
      .then((response) => {
        checkServerVersion(response);

        const { status } = response;

        if (status === 200) {
          return resolve(response.json());
        } if (status === 204) {
          return resolve();
        }

        const contentType = response.headers.get('content-type') || '';

        if (contentType.indexOf('application/json') >= 0) {
          return response.json().then((json) => {
            if (response.status === 401 && json && json instanceof Object) {
              if (!repeated && json.needToResetSessionId) {
                return resolve(resetSessionId()
                  .then(() => sendRequest(url, options, customHeader, true)));
              } if (json.redirectTo === 'passport') {
                Auth.redirectToPassport(true);
              } else if (json.redirectTo) {
                saveLocation();
                window.location.replace(json.redirectTo);
              }
            }

            return reject(json);
          });
        }

        if (contentType.indexOf('text/plain') >= 0) {
          return response.text().then((text) => reject({ code: response.code, message: text }));
        }

        return reject({ code: status, message: `Response is not OK: ${response.statusText}` });
      })
      .catch((response) => {
        if (response instanceof TypeError) {
          console.error('Network error', response);
          return reject({ code: 500, message: 'Network error', isNetworkError: true });
        }

        console.error('Uncaught exception while loading request', response);
        return reject({ code: 500, message: 'Uncaught exception while loading request' });
      });
  });
}

export function sendGet(url, urlParams) {
  if (urlParams !== undefined) {
    const search = formatSearch(urlParams);
    return sendRequest(`${url}?${search}`);
  }
  return sendRequest(url);
}

export function sendPost(url, body) {
  return sendRequest(url, {
    method: 'POST',
    body: JSON.stringify(body),
  }, false, false);
}

export function sendPut(url, body) {
  return sendRequest(url, {
    method: 'PUT',
    body: JSON.stringify(body),
  }, false, false);
}

export function sendDelete(url) {
  return sendRequest(url, {
    method: 'DELETE',
  }, false, false);
}

export function sendPatch(url, body) {
  return sendRequest(url, {
    method: 'PATCH',
    body: JSON.stringify(body),
  });
}
