import Auth from './Auth';
import { PROJECT_PERMISSIONS, PROJECT_PERMISSIONS_LIST } from './ProjectPermissions';
import { isCloudGpn, isCloudIL } from '../utils/env';

class ProjectAuth {
  static fromOld(project) {
    const { owner, acl, onlyAuthRead } = project;

    if (project.id === 'junk') {
      return new ProjectAuth(owner, false, PROJECT_PERMISSIONS_LIST);
    }

    if (isCloudGpn()) {
      return new ProjectAuth(owner, true, PROJECT_PERMISSIONS_LIST);
    }

    if (isCloudIL()) {
      return new ProjectAuth(owner, true, PROJECT_PERMISSIONS_LIST);
    }

    const login = Auth.state.login();
    const aclEntries = acl.filter((entry) => entry.login === login);

    let projectPermissions;
    if (aclEntries.length === 0) {
      projectPermissions = [];
    } else {
      projectPermissions = aclEntries[0].permissions;
    }

    return new ProjectAuth(owner, onlyAuthRead, projectPermissions);
  }

  static from(project, permissions) {
    const permissionsList = [];
    if (permissions.canRead) {
      permissionsList.push(PROJECT_PERMISSIONS.READ);
    }
    if (permissions.canUpdate) {
      permissionsList.push(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    }
    if (permissions.canDelete) {
      permissionsList.push(PROJECT_PERMISSIONS.CONFIG_DELETE);
    }

    const isOwner = Auth.state.login() === project.owner;

    return new ProjectAuth(permissions.authorizationType, isOwner, permissionsList);
  }

  constructor(authorizationType, isOwner, projectPermissions) {
    this._authorizationType = authorizationType;
    this._isOwner = isOwner;
    this._projectPermissions = projectPermissions;
  }

  isOwner() {
    return this._isOwner;
  }

  getAuthorizationType() {
    return this._authorizationType;
  }

  isAuthorizedFor(projectPermission) {
    return this._projectPermissions.includes(projectPermission || PROJECT_PERMISSIONS.READ);
  }
}

export default ProjectAuth;
