import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import JSONPretty from 'react-json-pretty';
import 'react-json-pretty/src/JSONPretty.acai.css';

import { ENV_INSTANCE, getEnv } from '../../utils/env';

import './style.css';

function getApiHostname() {
  const { hostname } = window.location;
  if (hostname === 'localhost') {
    // return {hostname}:{port}
    return window.location.host;
  }

  const env = getEnv();
  if (env.instance === ENV_INSTANCE.SERVICE) {
    return hostname.replace('yandex-team.ru', 'yandex.net');
  }

  return hostname;
}

const SOLOMON_API_ENDPOINT = `https://${getApiHostname()}/api/v2`;

function curlExample(path, isNew, id) {
  const pathWithId = isNew ? path : `${path}/${id}`;

  const textBefore = `$ curl -X ${isNew ? 'POST' : 'PUT'} \\
      -H 'Content-Type: application/json' \\
      -H 'Accept: application/json' \\
      -H 'Authorization: OAuth `;

  const tokenLinkTitle = '<YOUR_TOKEN>';

  const tokenLinkUrl = 'https://oauth.yandex-team.ru/authorize?response_type=token&client_id=1c0c37b3488143ff8ce570adb66b9dfa';

  const textAfter = `' \\
      -d @request.json \\
      ${SOLOMON_API_ENDPOINT}${pathWithId}`;

  return (
    <pre className="api-example-toggle-parent">
      {textBefore}
      <a href={tokenLinkUrl}>{tokenLinkTitle}</a>
      {textAfter}
    </pre>
  );
}

class ApiCallExample extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { editMode: false };
  }

  onChange = (event) => {
    event.preventDefault();
    let newState;
    try {
      newState = JSON.parse(event.target.value);
    } catch (e) {
      newState = null;
    }
    if (newState !== null) {
      this.props.onChange(newState);
    }
  };

  toggleEditMode = (event) => {
    event.preventDefault();
    this.setState({ editMode: !this.state.editMode });
  };

  toggleVisibility = (event) => {
    event.preventDefault();
    if (this.props.onChangeVisibility) {
      this.props.onChangeVisibility(!this.props.visible);
    }
  };

  render() {
    const {
      path, value, visible, isNew,
    } = this.props;
    const { editMode } = this.state;

    if (!visible && this.props.onChangeVisibility) {
      return (
        <div>
          <a role="button" tabIndex={-1} onClick={this.toggleVisibility}>
            &larr; Show API call example
          </a>
        </div>
      );
    }

    return (
      <div>
        <div className="api-example-toggle-parent">
          {this.props.onChangeVisibility && (
            <p>
              <a role="button" tabIndex={-1} onClick={this.toggleVisibility}>
                Hide API call example &rarr;
              </a>
            </p>
          )}
          {!editMode && <JSONPretty json={value} />}
          {editMode
            && (
            <textarea
              className="form-control api-example-json-editor" rows="25"
              defaultValue={JSON.stringify(value, null, 2)}
              onChange={this.onChange}
            />
            )}
          <span
            className="badge api-example-token-toggle"
            onClick={this.toggleEditMode} role="presentation"
          >
            {editMode ? 'close' : 'edit this JSON'}
          </span>
        </div>
        <span className="help-block">
          You can make same action by sending above JSON to Solomon API.
          Assume that you save it into request.json:
        </span>
        <pre className="api-example-toggle-parent">
          {curlExample(path, isNew, value.id)}
        </pre>
      </div>
    );
  }
}

ApiCallExample.propTypes = {
  path: PropTypes.string.isRequired,
  value: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  visible: PropTypes.bool,
  onChange: PropTypes.func.isRequired,
  onChangeVisibility: PropTypes.func,
};

ApiCallExample.defaultProps = {
  visible: false,
  onChangeVisibility: null,
};

export default ApiCallExample;
