import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

const ENTITY_TYPE_SHAPE = PropTypes.oneOf(['cluster', 'service']);

class AssociatedEntitiesTable extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { limit: props.limit };
  }

  onShowMoreClick = (event) => {
    event.preventDefault();
    this.setState({ limit: -1 });
  };

  onShowLessClick = (event) => {
    event.preventDefault();
    this.setState({ limit: this.props.limit });
  };

  render() {
    const {
      projectId, values, type, active, onAdd, onToggle, onDelete,
    } = this.props;
    const { limit } = this.state;
    const needLimit = (limit !== -1 && limit < values.length);
    const wasLimited = (limit !== this.props.limit);
    const limitedValues = needLimit ? values.slice(0, limit) : values;
    const trs = limitedValues.map((value, index) => (
      <tr>
        <td>{index + 1}</td>
        <td>
          <Link to={`/admin/projects/${projectId}/${type}s/${value.id}`}>
            {value.name || value.id}
          </Link>
        </td>
        <td>
          <Link to={`/admin/projects/${projectId}/shards/${value.shardId}`}>
            {value.shardId}
          </Link>
        </td>
        <td className="actions">
          {onToggle ? (
            <button type="button" onClick={() => onToggle(value)}>
              <i className={`glyphicon glyphicon-${active ? 'pause' : 'play'}`} />
            </button>
          ) : null}
          {onDelete ? (
            <button type="button" onClick={() => onDelete(value)}>
              <i className="glyphicon glyphicon-remove" />
            </button>
          ) : null }
        </td>
      </tr>
    ));

    let showMoreTr = null;

    if (needLimit) {
      showMoreTr = (
        <tr>
          <td colSpan={3}>
            <a href="#" onClick={this.onShowMoreClick}>
              Show
              {' '}
              {values.length - limit}
              {' '}
              more...
            </a>
          </td>
        </tr>
      );
    } else if (wasLimited) {
      showMoreTr = (
        <tr>
          <td colSpan={3}>
            <a href="#" onClick={this.onShowLessClick}>Show less...</a>
          </td>
        </tr>
      );
    }

    const addButton = (
      <button type="button" className="btn btn-default" onClick={onAdd}>
        <i className="glyphicon glyphicon-plus" />
      </button>
    );

    const addTr = onAdd ? (
      <tr className="action-row">
        <td colSpan={3}>
          {addButton}
        </td>
      </tr>
    ) : null;

    const tableClassName = 'table table-condensed';

    if (values.length === 0) {
      return onAdd ? addButton : (<span>No items found</span>);
    }

    const entityTitle = type === 'service' ? 'Service' : 'Cluster';

    return (
      <div className="table-responsive">
        <table className={tableClassName}>
          <thead>
            <tr>
              <th>#</th>
              <th>{entityTitle}</th>
              <th>Shard</th>
              <th />
            </tr>
          </thead>
          <tbody>
            {trs}
            {showMoreTr}
            {addTr}
          </tbody>
        </table>
      </div>
    );
  }
}

AssociatedEntitiesTable.propTypes = {
  projectId: PropTypes.string.isRequired,
  values: PropTypes.array.isRequired,
  type: ENTITY_TYPE_SHAPE.isRequired,
  limit: PropTypes.number.isRequired,
  active: PropTypes.bool.isRequired,
  onAdd: PropTypes.func,
  onToggle: PropTypes.func,
  onDelete: PropTypes.func,
};

AssociatedEntitiesTable.defaultProps = {
  onAdd: null,
  onToggle: null,
  onDelete: null,
};

export default AssociatedEntitiesTable;
