import React from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import capitalize from 'lodash/capitalize';

import './style.css';

function makeBreadcrumbItemLink(parts, index, params) {
  return parts.slice(0, index + 1).map((part) => {
    if (part.startsWith(':')) {
      return params[part.substr(1)];
    }
    return part;
  }).join('/');
}

function uncamelizePart(part) {
  return capitalize(part.replace(/[A-Z]/g, (letter) => ` ${letter.toLowerCase()}`)
    .replace('/^ /', ''));
}

class Breadcrumb extends React.PureComponent {
  render() {
    const { projectName, match } = this.props;

    const { path, params } = match;

    if (path === '*') {
      return null;
    }

    let linkPrefix;
    let fixedPath;

    if (path.startsWith('/admin/projects')) {
      fixedPath = path.substr('/admin/projects'.length);
      linkPrefix = '/admin/projects/';
    } else if (path.startsWith('/admin')) {
      fixedPath = path.substr('/admin'.length);
      linkPrefix = '/admin/';
    }

    const parts = fixedPath.split('/').filter((part) => !!part);

    return (
      <ol className="breadcrumb">
        <li>
          <Link to="/admin/projects">Home</Link>
        </li>
        {parts.map((part, index) => {
          let title;

          if (part.startsWith(':')) {
            title = params[part.substr(1)];
            if (part === ':projectId') {
              title = projectName;
            }
            if (part === ':alertId') {
              title = this.props.alertName;
            }
            if (part === ':channelId') {
              title = this.props.channelName;
            }
          } else {
            title = uncamelizePart(part);
          }

          let content;

          const active = index === parts.length - 1;

          if (!active) {
            content = (
              <Link to={`${linkPrefix}${makeBreadcrumbItemLink(parts, index, params)}`}>
                {title}
              </Link>
            );
          } else {
            content = title;
          }

          return (<li key={part} className={active ? 'active' : ''}>{content}</li>);
        })}
      </ol>
    );
  }
}

Breadcrumb.propTypes = {
  projectName: PropTypes.string.isRequired,
  alertName: PropTypes.string,
  channelName: PropTypes.string,
  match: PropTypes.object.isRequired,
};

Breadcrumb.defaultProps = {
  alertName: '',
  channelName: '',
};

export default connect(
  (state) => ({
    projectName: state.projectData.project.name || '',
    channelName: state.channel ? (state.channel.name || '') : '',
  }),
)(Breadcrumb);
