import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import FormInput from '../forms/FormInput';
import FormRadioGroup from '../../pages/graphs/FormRadioGroup';
import FormArea from '../forms/FormArea';

const DASHBOARD_TYPE_OPTIONS = [
  { value: 'IFRAME', title: 'Iframe' },
  { value: 'MARKDOWN', title: 'Markdown' },
];

class DashboardEditablePanel extends PureComponent {
  constructor(props) {
    super(props);
    this.state = props.panel;
  }

  componentDidMount() {
    this.titleInput.focus();
  }

  onInputChange = (event) => {
    event.preventDefault();
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  onTypeChange = (event) => {
    this.setState({ [event.target.name]: event.target.value });
  };

  onSubmitClick = () => {
    const state = { ...this.state };

    const type = state.type || 'IFRAME';

    if (type === 'IFRAME') {
      delete state.markdown;
    }

    if (type === 'MARKDOWN') {
      delete state.url;
    }

    this.props.onSubmit(this.props.index, state);
  };

  onDiscardClick = () => {
    this.props.onDiscard(this.props.index);
  };

  render() {
    const panel = this.state;

    const panelType = panel.type || 'IFRAME';

    const panelTitle = this.props.index < 0 ? 'New panel' : `Panel #${this.props.index + 1}`;

    return (
      <div className="panel panel-default">
        <div className="panel-heading">
          <div className="clearfix">
            <div className="pull-left">
              <div className="panel-title">
                {panelTitle}
              </div>
            </div>
            <div className="pull-right">
              <button type="button" className="btn btn-default" onClick={this.onSubmitClick}>
                <i className="glyphicon glyphicon-ok" />
              </button>
              &nbsp;
              <button type="button" className="btn btn-default" onClick={this.onDiscardClick}>
                <i className="glyphicon glyphicon-remove" />
              </button>
            </div>
          </div>
        </div>
        <div className="panel-body">
          <div className="form-horizontal">
            <FormInput
              type="text"
              name="title" label="Title" value={panel.title}
              placeholder="Title..."
              onChange={this.onInputChange}
              ref={(input) => { this.titleInput = input; }}
            />
            <FormInput
              type="text"
              name="subtitle" label="Subtitle" value={panel.subtitle}
              onChange={this.onInputChange}
            />
            <FormRadioGroup
              name="type"
              label="Type"
              options={DASHBOARD_TYPE_OPTIONS}
              value={panelType}
              onChange={this.onTypeChange}
            />
            {panelType === 'IFRAME' && (
              <FormArea
                type="text"
                rows={5}
                name="url" label="Url" value={panel.url}
                onChange={this.onInputChange}
              />
            )}
            {panelType === 'MARKDOWN' && (
              <FormArea
                type="text"
                rows={10}
                name="markdown" label="Markdown" value={panel.markdown}
                onChange={this.onInputChange}
              />
            )}
            <FormInput
              type="number"
              name="rowspan" label="Row span" value={panel.rowspan}
              onChange={this.onInputChange}
            />
            <FormInput
              type="number"
              name="colspan" label="Col span" value={panel.colspan}
              onChange={this.onInputChange}
            />
          </div>
        </div>
      </div>
    );
  }
}

DashboardEditablePanel.propTypes = {
  index: PropTypes.number.isRequired,
  panel: PropTypes.object,
  onSubmit: PropTypes.func.isRequired,
  onDiscard: PropTypes.func.isRequired,
};

DashboardEditablePanel.defaultProps = {
  panel: {
    title: '', url: '', colspan: 0, rowspan: 0,
  },
};

export default DashboardEditablePanel;
