/* eslint-disable jsx-a11y/no-static-element-interactions */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import './style.css';

const KEYS = {
  DOWN_ARROW: 40,
  UP_ARROW: 38,
  ENTER: 13,
  TAB: 9,
  ESC: 27,
};

class Datalist extends PureComponent {
  constructor(props) {
    super(props);
    this._mouseOver = false;
    this.state = {
      value: props.value || '',
      activeIndex: -1,
      opened: props.defaultOpened,
    };
  }

  componentDidMount() {
    if (this.props.defaultOpened && this._input) {
      this._input.focus();
    }
  }

  componentDidUpdate(prevProps) {
    if (this.state.opened && this._activeLi) {
      this._scrollDropdownToActiveItem();
    }
    if (this.props.value !== prevProps.value) {
      this.setState({ value: this.props.value || '' });
    }
  }

  handleChange = (event) => {
    this.setState({ value: event.target.value, opened: true, activeIndex: 0 });
    if (this.props.onChange) {
      this.props.onChange(event);
    }
  };

  handleItemSelect = (item) => {
    if (item.onClick) {
      item.onClick(item);
    } else {
      this.setState({ value: item.value, opened: false, activeIndex: -1 });
      if (this.props.onApply) {
        this.props.onApply(item.value);
      } else if (this.props.onChange) {
        this.props.onChange({ target: { value: item.value, name: this.props.name } });
      }
    }
  };

  handleKeyDown = (event) => {
    const { activeIndex } = this.state;
    const { datalist } = this.props;

    if (datalist.length === 0 || event.defaultPrevented) {
      this._hideDropdown();
      return;
    }

    switch (event.keyCode) {
      case KEYS.DOWN_ARROW: {
        event.preventDefault();
        const nextIndex = (activeIndex + 1) % datalist.length;
        this.setState({ activeIndex: nextIndex, opened: true });
        break;
      }

      case KEYS.UP_ARROW: {
        event.preventDefault();
        const nextIndex = activeIndex > 0 ? activeIndex - 1 : datalist.length - 1;
        this.setState({ activeIndex: nextIndex, opened: true });
        break;
      }

      case KEYS.ENTER: {
        if (activeIndex >= 0 && activeIndex < datalist.length) {
          event.preventDefault();
          const item = datalist[activeIndex];
          this.handleItemSelect(item);
        }
        break;
      }

      case KEYS.TAB: {
        this.setState({ opened: false, activeIndex: -1 });
        break;
      }

      case KEYS.ESC: {
        event.preventDefault();
        event.stopPropagation();
        this._hideDropdown();
        if (this.props.onBlur) {
          this.props.onBlur();
        }
        break;
      }

      default:
        break;
    }
  };

  handleMouseDown = (event) => {
    event.preventDefault();
    this._input.focus();
    if (!this.state.opened && !this.props.disabled) {
      this.setState({ opened: true, activeIndex: 0 });
    }
  };

  handleBlur = () => {
    if (!this._mouseOver) {
      this.setState({ opened: false, activeIndex: -1 });
      if (this.props.onBlur) {
        this.props.onBlur();
      }
    }
  };

  focus() {
    this._input.focus();
  }

  _hideDropdown() {
    this.setState({ opened: false, activeIndex: -1 });
  }

  _scrollDropdownToActiveItem() {
    if (!this._activeLi) {
      return;
    }

    const liScrollTop = this._activeLi.offsetTop;
    const liScrollBottom = liScrollTop + this._activeLi.clientHeight;
    const dropdownScrollTop = this._dropdown.scrollTop;
    const dropdownHeight = this._dropdown.clientHeight;

    let newDropdownScrollTop;
    if (liScrollTop < dropdownScrollTop) {
      newDropdownScrollTop = liScrollTop;
    } else if (liScrollBottom > dropdownScrollTop + dropdownHeight) {
      newDropdownScrollTop = liScrollBottom - dropdownHeight;
    } else {
      newDropdownScrollTop = dropdownScrollTop;
    }
    this._dropdown.scrollTop = newDropdownScrollTop;
  }

  render() {
    const {
      name, placeholder, disabled, datalist,
    } = this.props;
    const { value, opened, activeIndex } = this.state;

    const showDropdown = opened && datalist.length > 0;

    return (
      <div
        className={`datalist dropdown${showDropdown ? ' open' : ''}`}
        onMouseDown={this.handleMouseDown}
        onMouseEnter={this.handleMouseEnter}
        onMouseLeave={this.handleMouseLeave}
      >
        <input
          name={name}
          type="text"
          className={`${this.props.size === 'xs' ? ' input-xs' : 'form-control'}`}
          value={value}
          autoComplete="off"
          placeholder={placeholder}
          disabled={disabled}
          onChange={this.handleChange}
          onFocus={this.handleFocus}
          onBlur={this.handleBlur}
          onKeyDown={this.handleKeyDown}
          ref={(input) => { this._input = input; }}
        />
        <ul
          role="menu" className="dropdown-menu"
          ref={(dropdown) => { this._dropdown = dropdown; }}
        >
          {datalist.map((item, index) => (
            <li
              role="presentation"
              className={index === activeIndex ? 'active' : null}
              key={item.value}
              ref={(li) => { if (index === activeIndex) { this._activeLi = li; } }}
            >
              <a
                role="menuItem"
                tabIndex={index}
                onClick={() => this.handleItemSelect(item)}
              >
                {item.content || item.value}
              </a>
            </li>
          ))}
        </ul>
      </div>
    );
  }
}

Datalist.propTypes = {
  name: PropTypes.string,
  value: PropTypes.string,
  placeholder: PropTypes.string,
  size: PropTypes.string,
  datalist: PropTypes.array,
  onChange: PropTypes.func,
  onApply: PropTypes.func,
  onBlur: PropTypes.func,
  disabled: PropTypes.bool,
  defaultOpened: PropTypes.bool,
};

Datalist.defaultProps = {
  name: '',
  value: '',
  placeholder: '',
  size: '',
  datalist: [],
  onChange: null,
  onApply: null,
  onBlur: null,
  disabled: false,
  defaultOpened: false,
};

export default Datalist;
