import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import FormInput from '../forms/FormInput';
import FormElement from '../forms/FormElement';
import RadioButtons from '../RadioButtons';
import FormCheckbox from '../forms/FormCheckbox';
import FormArea from '../forms/FormArea';
import * as SETTINGS from '../../pages/graphs/settings';
import FormRadioGroup from '../../pages/graphs/FormRadioGroup';

const TRANSFORM_OPTIONS = [
  { value: 'NONE', title: 'None' },
  { value: 'DIFFERENTIATE', title: 'Differentiate (without negative)' },
  { value: 'DIFFERENTIATE_WITH_NEGATIVE', title: 'Differentiate (with negative)' },
  { value: 'INTEGRATE', title: 'Integrate' },
];

class EditableElement extends PureComponent {
  constructor(props) {
    super(props);

    const selectorsAsText = (props.value.selectors || []).map((s) => `${s.name}=${s.value}`).join(',');
    this.state = { value: props.value, selectorsAsText };
  }

  componentDidMount() {
    this.titleInput.focus();
  }

  onInputChange = (event) => {
    const newValue = { ...this.state.value };
    const { name, value } = event.target;
    newValue[name] = value;
    this.setState({ value: newValue });
  };

  onTypeChange = (type) => {
    const value = { ...this.state.value };
    value.type = type;
    this.setState({ value });
  };

  onSelectorsAsTextChange = (event) => {
    const selectorsAsText = event.target.value;
    const selectorsSplitted = selectorsAsText.split(/[,&]/);
    const selectors = selectorsSplitted.map((s) => {
      const split = s.split('=');
      if (split.length !== 2) {
        return null;
      }
      return { name: split[0], value: split[1] };
    });
    const newState = { selectorsAsText };
    if (selectors.every((s) => !!s)) {
      newState.value = { ...this.state.value };
      newState.value.selectors = selectors;
    }
    this.setState(newState);
  };

  onBoolChange = (event) => {
    const value = { ...this.state.value };
    const { name, checked } = event.target;
    value[name] = checked;
    this.setState({ value });
  };

  onBoolSelectChange = (event) => {
    const value = { ...this.state.value };

    const { name } = event.target;

    let boolValue;
    switch (event.target.value) {
      case 'true':
        boolValue = true;
        break;
      case 'false':
        boolValue = false;
        break;
      default:
        boolValue = null;
    }

    value[name] = boolValue;

    this.setState({ value });
  };

  onYaxisChange = (yaxis) => {
    const value = { ...this.state.value, yaxis };
    this.setState({ value });
  };

  onSubmitClick = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.props.index, this.state.value);
  };

  onDiscardClick = (event) => {
    event.preventDefault();
    this.props.onDiscard(this.props.index);
  };

  render() {
    const { value, selectorsAsText } = this.state;
    const type = value.type || 'SELECTORS';
    const elementTitle = this.props.index < 0 ? 'New element' : `Element #${this.props.index + 1}`;

    return (
      <div className="panel panel-default">
        <div className="panel-heading">
          <div className="clearfix">
            <div className="pull-left">
              <div className="panel-title">
                {elementTitle}
              </div>
            </div>
            <div className="pull-right">
              <button type="button" className="btn btn-xs btn-default" onClick={this.onSubmitClick}>
                <i className="glyphicon glyphicon-ok" />
              </button>
              &nbsp;
              <button type="button" className="btn btn-xs btn-default" onClick={this.onDiscardClick}>
                <i className="glyphicon glyphicon-remove" />
              </button>
            </div>
          </div>
        </div>
        <div className="panel-body">
          <div className="form-horizontal">
            <FormInput
              type="text"
              name="title"
              label="Title"
              value={value.title}
              placeholder="Title..."
              onChange={this.onInputChange}
              ref={(input) => { this.titleInput = input; }}
            />
            <FormElement label="Type">
              <RadioButtons
                choices={['SELECTORS', 'EXPRESSION']}
                labels={['Selectors', 'Expression']}
                selected={type}
                onSelect={this.onTypeChange}
              />
            </FormElement>
            {type === 'SELECTORS' && (
              <FormInput
                type="text"
                name="selectors"
                label="Selectors"
                value={selectorsAsText}
                onChange={this.onSelectorsAsTextChange}
              />
            )}
            {type === 'SELECTORS' && (
              <FormInput
                type="text"
                name="link"
                label="Link"
                value={value.link}
                onChange={this.onInputChange}
              />
            )}
            {type === 'EXPRESSION' && (
              <FormArea
                type="text"
                name="expression"
                label="Expression"
                rows={10}
                value={value.expression}
                onChange={this.onInputChange}
              />
            )}
            <FormInput
              type="text"
              name="color"
              label="Color"
              value={value.color}
              onChange={this.onInputChange}
            />
            <FormRadioGroup
              name="area" label="Area"
              value={String(value.area)}
              defaultValue="null"
              options={SETTINGS.STACK_OPTIONS}
              onChange={this.onBoolSelectChange}
            />
            <FormInput
              type="text"
              name="stack"
              label="Stack"
              value={value.stack}
              onChange={this.onInputChange}
            />
            <FormCheckbox
              name="down"
              label="Growing down stack"
              value={value.down}
              onChange={this.onBoolChange}
            />
            <FormElement label="Y-axis position">
              <RadioButtons
                choices={['LEFT', 'RIGHT']}
                labels={['Left', 'Right']}
                selected={value.yaxis || 'LEFT'}
                onSelect={this.onYaxisChange}
              />
            </FormElement>
            <FormElement label="Transform">
              <select
                className="form-control"
                name="transform"
                value={value.transform}
                onChange={this.onInputChange}
              >
                {TRANSFORM_OPTIONS.map((option, index) => (
                  // eslint-disable-next-line react/no-array-index-key
                  <option key={index} value={option.value}>
                    {option.title}
                  </option>
                ))}
              </select>
            </FormElement>
          </div>
        </div>
      </div>
    );
  }
}

EditableElement.propTypes = {
  index: PropTypes.number.isRequired,
  value: PropTypes.object,
  onSubmit: PropTypes.func.isRequired,
  onDiscard: PropTypes.func.isRequired,
};

EditableElement.defaultProps = {
  value: {
    title: '',
    type: 'SELECTORS',
    selectors: [],
    expression: '',
    area: null,
    color: '',
    stack: '',
    down: false,
    yaxis: 'LEFT',
    transform: 'NONE',
  },
};

export default EditableElement;
