import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import EditableElement from './EditableElement';
import ReadonlyElement from './ReadonlyElement';

class GraphElementEditor extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { elementEditIndex: -1 };
  }

  onSubmitElement = (index, element) => {
    const newElements = [...this.props.elements];
    if (index === -2) {
      newElements.push(element);
    } else {
      newElements.splice(index, 1, element);
    }
    this.props.onChange(newElements);
    this.setState({ elementEditIndex: -1 });
  };

  onDiscardElement = () => {
    this.setState({ elementEditIndex: -1 });
  };

  onEditElement = (index) => {
    this.setState({ elementEditIndex: index });
  };

  onDeleteElement = (index) => {
    const newElements = [...this.props.elements];
    newElements.splice(index, 1);
    this.props.onChange(newElements);
    this.setState({ elementEditIndex: -1 });
  };

  onAddElement = () => {
    this.setState({ elementEditIndex: -2 });
  };

  render() {
    const { elements, readOnly } = this.props;
    const { elementEditIndex } = this.state;
    return (
      <div>
        <div>
          {elements.map((el, index) => {
            if (index === elementEditIndex) {
              return (
                <EditableElement
                  index={index}
                  // eslint-disable-next-line react/no-array-index-key
                  key={index}
                  value={el}
                  onSubmit={this.onSubmitElement}
                  onDiscard={this.onDiscardElement}
                />
              );
            }
            return (
              <ReadonlyElement
                index={index}
                // eslint-disable-next-line react/no-array-index-key
                key={index}
                value={el}
                onEdit={this.onEditElement}
                onDelete={this.onDeleteElement}
                readOnly={readOnly}
              />
            );
          })}
          {elementEditIndex === -2 && (
            <EditableElement
              index={elementEditIndex}
              onSubmit={this.onSubmitElement}
              onDiscard={this.onDiscardElement}
            />
          )}
        </div>
        {!readOnly && (
        <div>
          <button type="button" className="btn btn-default" onClick={this.onAddElement}>
            <i className="glyphicon glyphicon-plus" />
          </button>
        </div>
        )}
      </div>
    );
  }
}

GraphElementEditor.propTypes = {
  elements: PropTypes.array,
  readOnly: PropTypes.bool,
  onChange: PropTypes.func,
};

GraphElementEditor.defaultProps = {
  elements: [],
  readOnly: false,
  onChange: () => {},
};

export default GraphElementEditor;
