/* eslint-disable no-continue */
import React from 'react';
import PropTypes from 'prop-types';
import MenuItem from 'react-bootstrap/es/MenuItem';
import NavDropdown from 'react-bootstrap/es/NavDropdown';
import { LinkContainer } from 'react-router-bootstrap';

import { parseSearch } from '../../utils/url';
import { ENVIRONMENTS_LIST, getEnv, isPreOrProdIntEnv } from '../../utils/env';
import BrowserUtils from '../../utils/BrowserUtils';
import UrlUtils from '../../utils/url/UrlUtils';
import UserLinksBasic from '../../utils/UserLinksBasic';
import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';
import {
  getMonitoringUiHost,
} from '../../pages/old/utils/graphToExpressionTransformations';

function createTicketUrl() {
  const stUrl = 'https://st.yandex-team.ru/createTicket?queue=MONSUPPORT';

  return `${stUrl}`;
}

function createProjectMenuEditItem(project, title) {
  return (
    <LinkContainer key="admin-project-menu-item" to={`/admin/projects/${project}/menu`}>
      <MenuItem>{title}</MenuItem>
    </LinkContainer>
  );
}

const isNotActiveLambda = () => false;

function createAdminMenuItems(project, search, shardId) {
  const adminMenuItems = [];

  adminMenuItems.push((
    <LinkContainer key="admin-item" to="/admin" isActive={isNotActiveLambda}>
      <MenuItem>Admin</MenuItem>
    </LinkContainer>
  ));

  if (project) {
    adminMenuItems.push((
      <LinkContainer
        key="admin-project-item"
        to={`/admin/projects/${project}`}
      >
        <MenuItem>Edit project</MenuItem>
      </LinkContainer>
    ));

    adminMenuItems.push(createProjectMenuEditItem(project, 'Edit project menu'));

    if (shardId) {
      adminMenuItems.push((
        <LinkContainer
          key="admin-shard-item"
          to={`/admin/projects/${project}/shards/${shardId}`}
        >
          <MenuItem>Edit shard</MenuItem>
        </LinkContainer>
      ));
    }
  }

  return adminMenuItems;
}

function createConfigureMenuItems(project) {
  if (project) {
    return [
      <LinkContainer key="graphs-item" to={`/admin/projects/${project}/graphs`}>
        <MenuItem>Configure graphs</MenuItem>
      </LinkContainer>,
      <LinkContainer key="dashboards-item" to={`/admin/projects/${project}/dashboards`}>
        <MenuItem>Configure dashboards</MenuItem>
      </LinkContainer>,
      <LinkContainer key="alerts-item" to={`/admin/projects/${project}/alerts`}>
        <MenuItem>Configure alerts</MenuItem>
      </LinkContainer>,
    ];
  }

  return [];
}

function createEditMenuItems(project, graph, dashboard) {
  if (graph && graph !== 'auto') {
    return [
      <LinkContainer key="graph-edit-item" to={`/admin/projects/${project}/graphs/${graph}/edit`}>
        <MenuItem key="edit-item">Edit graph</MenuItem>
      </LinkContainer>,
    ];
  }

  if (dashboard) {
    return [
      <LinkContainer key="dash-edit-item" to={`/admin/projects/${project}/dashboards/${dashboard}/edit`}>
        <MenuItem key="edit-item">Edit dashboard</MenuItem>
      </LinkContainer>,
    ];
  }

  return [];
}

function createForceDcMenuItems(shardDcs) {
  const forceMenuItems = [];

  if (!shardDcs || shardDcs.length === 0) {
    return forceMenuItems;
  }

  // eslint-disable-next-line max-len
  const currentForceCluster = BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_CLUSTER_PARAM)
    || BrowserUtils.getLogicalQueryArgOrEmpty(UserLinksBasic.FORCE_HOST_PARAM);

  const url = UrlUtils.removeQueryArgs(
    BrowserUtils.getLogicalUrlAfterHost(),
    UserLinksBasic.FORCE_HOST_PARAM,
  );

  for (let i = 0; i < shardDcs.length; ++i) {
    let forceParamValue = shardDcs[i];

    // safety net
    if (!forceParamValue) {
      continue;
    }

    forceParamValue = forceParamValue.replace(/\\..*/, '');
    const href = UrlUtils.updateParameter(url, UserLinksBasic.FORCE_CLUSTER_PARAM, forceParamValue);
    const title = `${UserLinksBasic.FORCE_CLUSTER_PARAM}=${forceParamValue}`;
    if (forceParamValue === currentForceCluster) {
      continue;
    }
    forceMenuItems.push((
      <MenuItem
        key={`force-${forceParamValue}-item`}
        href={href}
      >
        {title}
      </MenuItem>
    ));
  }

  if (currentForceCluster) {
    const href = UrlUtils.removeQueryArgs(
      url,
      UserLinksBasic.FORCE_CLUSTER_PARAM,
    );
    const title = `do not ${UserLinksBasic.FORCE_CLUSTER_PARAM}`;
    forceMenuItems.push((
      <MenuItem
        key="do-not-force-item"
        href={href}
      >
        {title}
      </MenuItem>
    ));
  }

  return forceMenuItems;
}

function createDevMenuItems(shardDcs) {
  const env = getEnv();

  const host = window.location.host.toString();

  const devMenuItems = [];

  if (Auth.hasRole(ADMIN)) {
    devMenuItems.push(<MenuItem href="/staffOnly" key="dev-staffOnly">Staff only</MenuItem>);
  }

  for (let i = 0; i < ENVIRONMENTS_LIST.length; ++i) {
    const curEnv = ENVIRONMENTS_LIST[i];
    if (curEnv.instance === env.instance && curEnv.type !== env.type) {
      for (let j = 0; j < curEnv.hosts.length; ++j) {
        const curHost = curEnv.hosts[j];

        if (host !== curHost) {
          const schema = host.startsWith('localhost') ? 'http' : 'https';
          const url = `${schema}://${curHost}${window.location.pathname}${window.location.search}`;
          devMenuItems.push(
            <MenuItem href={url} key={`dev-${i}`}>
              This @
              {curHost}
            </MenuItem>,
          );
          break;
        }
      }
    }
  }

  devMenuItems.push(...createForceDcMenuItems(shardDcs));

  return devMenuItems;
}

// eslint-disable-next-line object-curly-newline
const CogMenu = ({ search, shardId, shardDcs }) => {
  const parsedSearch = parseSearch(search);

  const project = parsedSearch.get('project') || '';
  const graph = parsedSearch.get('graph') || '';
  const dashboard = parsedSearch.get('dashboard') || '';

  const stUrl = createTicketUrl();
  const adminItems = createAdminMenuItems(project, parsedSearch, shardId);
  const configureMenuItems = createConfigureMenuItems(project);
  const editMenuItems = createEditMenuItems(project, graph, dashboard);
  const devMenuItems = createDevMenuItems(shardDcs);

  let monitoringLink = '';

  if (project) {
    monitoringLink += `https://${getMonitoringUiHost()}/projects/${project}?utm_source=solomon_menu_view`;
  }

  if (dashboard !== '') {
    monitoringLink = `${window.location.href}&utm_source=solomon_menu_view&__force_new=1`;
  }

  if (graph !== '') {
    monitoringLink = `${window.location.href}&utm_source=solomon_menu_view&__force_new=1`;
  }

  return (
    <>
      <NavDropdown
        title={<span className="glyphicon glyphicon-cog" />}
        id="cog-menu"
        noCaret
      >
        {isPreOrProdIntEnv() && (
          <>
            <MenuItem href={monitoringLink} target="_blank">
              <strong>New: Monitoring UI</strong>
            </MenuItem>
            <MenuItem divider />
          </>
        )}
        <MenuItem
          href="https://clubs.at.yandex-team.ru/solomon/"
          target="_blank"
        >
          Atoushka
        </MenuItem>
        <MenuItem href="mailto:solomon@yandex-team.ru" target="_blank">
          Mailing list
        </MenuItem>
        <MenuItem href={stUrl} target="_blank">
          Report a problem
        </MenuItem>
        {adminItems.length > 0 && <MenuItem divider />}
        {adminItems}
        {configureMenuItems.length > 0 && <MenuItem divider />}
        {configureMenuItems}
        {editMenuItems.length > 0 && <MenuItem divider />}
        {editMenuItems}
        <MenuItem divider />
        <LinkContainer key="preferences-item" to="/admin/preferences">
          <MenuItem key="preferences-item">Preferences</MenuItem>
        </LinkContainer>
        {devMenuItems}
      </NavDropdown>
    </>
  );
};

CogMenu.propTypes = {
  search: PropTypes.string.isRequired,
  shardId: PropTypes.string,
  shardDcs: PropTypes.array,
};

CogMenu.defaultProps = {
  shardId: '',
  shardDcs: [],
};

export default CogMenu;
