import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import Tooltip from 'react-bootstrap/es/Tooltip';
import OverlayTrigger from 'react-bootstrap/es/OverlayTrigger';

import addWildcardLinks from '../../utils/wildcards/Wildcards';
import LabelValidator from './LabelValidator';

import './style.css';

class LabelGroup extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { showMore: true };
  }

  onShowMoreClick = (event) => {
    event.preventDefault();
    this.setState({ showMore: false });
  };

  onShowLessClick = (event) => {
    event.preventDefault();
    this.setState({ showMore: true });
    this._headerEl.scrollIntoView();
  };

  render() {
    const {
      name, values,
      absent, truncated, limit,
      canGroupByLabel, canCollapse,
      enableValidation,
    } = this.props;

    const needLimit = limit > 0 && values.length > limit;
    const showLimit = this.state.showMore && needLimit;
    const hideLimit = !this.state.showMore && needLimit;

    const limitedValues = showLimit ? values.slice(0, limit) : values;

    let labelHeader;

    if (canGroupByLabel) {
      const href = this.props.formatUrl(name, '*');
      labelHeader = <Link to={href}>{name}</Link>;
    } else {
      labelHeader = name;
    }

    if (enableValidation) {
      const validationResult = LabelValidator.validateKey(name);

      if (validationResult) {
        const tooltip = (
          <Tooltip>
            Label key is not valid:
            {' '}
            {validationResult}
          </Tooltip>
        );

        labelHeader = (
          <div className="invalid-label-header">
            <h4>
              <span className="invalid-label-header__title">
                {labelHeader}
              </span>
            </h4>
            <OverlayTrigger placement="right" overlay={tooltip}>
              <small className="invalid-label-header__tooltip">
                <span className="glyphicon glyphicon-info-sign" />
              </small>
            </OverlayTrigger>
          </div>
        );
      } else {
        labelHeader = <h4>{labelHeader}</h4>;
      }
    } else {
      labelHeader = <h4>{labelHeader}</h4>;
    }

    let valuesWithWildcards = limitedValues;

    if (!enableValidation && canGroupByLabel && name !== 'cluster' && name !== 'service') {
      try {
        valuesWithWildcards = addWildcardLinks(limitedValues);
      } catch (e) {
        console.error('Failed to build wildcards', e);
      }
    }

    return (
      <div className="label-group">
        <div className="label-group__header" ref={(el) => { this._headerEl = el; }}>
          {labelHeader}
        </div>
        <ul className="label-group__values">
          {absent && !enableValidation && (
            <li>
              <Link to={this.props.formatUrl(name, '-')}>- (without label)</Link>
            </li>
          )}
          {valuesWithWildcards.map((labelValue, index) => {
            const href = this.props.formatUrl(name, labelValue);
            let content = (
              <Link to={href}>
                {labelValue}
              </Link>
            );

            if (enableValidation) {
              const validationResult = LabelValidator.validateValue(labelValue);

              if (validationResult) {
                const tooltip = (
                  <Tooltip>
                    {validationResult}
                  </Tooltip>
                );

                const tooltipTrigger = (
                  <OverlayTrigger placement="right" overlay={tooltip}>
                    <small>
                      <span className="glyphicon glyphicon-info-sign" />
                    </small>
                  </OverlayTrigger>
                );

                content = (
                  <span>
                    {content}
                    &nbsp;
                    {tooltipTrigger}
                  </span>
                );
              }
            }

            return (
              // eslint-disable-next-line react/no-array-index-key
              <li key={index}>
                {content}
              </li>
            );
          })}
          {!showLimit && truncated && (
            <li>...</li>
          )}
          {showLimit && (
            <li>
              <a href="#" onClick={this.onShowMoreClick}>
                {truncated ? `Show first ${values.length} values...` : `Show ${values.length - limit} more...`}
              </a>
            </li>
          )}
          {hideLimit && canCollapse && (
            <li>
              <a href="#" onClick={this.onShowLessClick}>
                Show less...
              </a>
            </li>
          )}
        </ul>
      </div>
    );
  }
}

LabelGroup.propTypes = {
  name: PropTypes.string.isRequired,
  values: PropTypes.array.isRequired,
  absent: PropTypes.bool.isRequired,
  truncated: PropTypes.bool.isRequired,
  enableValidation: PropTypes.bool,
  limit: PropTypes.number,
  canGroupByLabel: PropTypes.bool,
  canCollapse: PropTypes.bool,
  formatUrl: PropTypes.func.isRequired,
};

LabelGroup.defaultProps = {
  limit: 10,
  canGroupByLabel: true,
  canCollapse: true,
  enableValidation: false,
};

export default LabelGroup;
