/* eslint-disable react/prop-types */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import { Link } from 'react-router-dom';

import UserLinksBasic from '../../utils/UserLinksBasic';

import './style.css';

const DISTANCE_FROM_SELECTPICKER_BOTTOM_AND_CLIENT_BOTTOM = 20;

function isTargetInsideElement(target, element) {
  let tempTarget = target;

  const DOCUMENT_NODE_TYPE = 9;

  // eslint-disable-next-line no-constant-condition
  while (true) {
    if (!tempTarget) {
      return false;
    }

    if (tempTarget.nodeType === DOCUMENT_NODE_TYPE) {
      return false;
    }

    if (tempTarget === element) {
      return true;
    }

    tempTarget = tempTarget.parentElement;
  }
}

const LabelSelectorOptionProps = PropTypes.shape({
  value: PropTypes.string,
  fullPath: PropTypes.string.isRequired,
  selected: PropTypes.bool.isRequired,
});

const LabelSelectorStateProps = PropTypes.shape({
  loading: PropTypes.bool,
  data: PropTypes.arrayOf(LabelSelectorOptionProps),
  error: PropTypes.string,
});

class LabelSelector extends PureComponent {
  constructor(props) {
    super(props);

    let activeIndex = -1;

    for (let i = 0; i < props.options.length; ++i) {
      const option = props.options[i];
      if (option.selected) {
        activeIndex = i;
        break;
      }
    }

    this.state = {
      opened: props.defaultOpened,
      filter: props.value,
      activeIndex,
      showAllAsync: props.showAllValues,
    };
  }

  componentDidMount() {
    document.addEventListener('mousedown', this.handleDocumentClick, true);
    if (this.state.opened) {
      this._searchbox.focus();
      this.updateDropdownHeight();
      this.loadOptions(true);
    }
  }

  componentDidUpdate() {
    if (this.state.opened && this._activeLi) {
      this.scrollToActiveItem();
    }
  }

  componentWillUnmount() {
    document.removeEventListener('mousedown', this.handleDocumentClick, true);
  }

  updateDropdownHeight = () => {
    const innerMenuRect = this._innerMenu.getBoundingClientRect();
    const outerMenuRect = this._outerMenu.getBoundingClientRect();

    const menuMargin = outerMenuRect.height - innerMenuRect.height;

    const dropdownMenuTop = outerMenuRect.top;
    const windowHeight = window.innerHeight;
    const height = windowHeight - dropdownMenuTop
      - DISTANCE_FROM_SELECTPICKER_BOTTOM_AND_CLIENT_BOTTOM
      - menuMargin;

    this._innerMenu.style.maxHeight = `${height}px`;
  };

  handleDocumentClick = (event) => {
    if (!isTargetInsideElement(event.target, this._selectpicker)) {
      this.hideDropdown();
    }
  };

  handleDropdownToggle = () => {
    const newDropdownShown = !this.state.opened;

    if (newDropdownShown) {
      this.showDropdown();
    } else {
      this.hideDropdown();
    }
  };

  handleInputChange = (event) => {
    event.preventDefault();
    this.setState({ filter: event.target.value }, () => {
      this.loadOptions();
    });
  };

  handleInputKeyDown = (event) => {
    const searchValue = event.target.value;

    const { activeIndex } = this.state;
    const { options } = this.props;

    switch (event.keyCode) {
      case 40: {
        event.preventDefault();
        const nextIndex = (activeIndex + 1) % options.length;
        this.setState({ activeIndex: nextIndex, opened: true });
        break;
      }

      case 38: {
        event.preventDefault();
        const nextIndex = activeIndex > 0 ? activeIndex - 1 : options.length - 1;
        this.setState({ activeIndex: nextIndex, opened: true });
        break;
      }

      case 13: {
        if (activeIndex >= 0 && activeIndex < options.length) {
          event.preventDefault();
          this.hideDropdown();
          const option = options[activeIndex];
          this.context.router.history.push(option.fullPath);
        } else if (searchValue) {
          this.hideDropdown();
          this.props.onSelect(searchValue);
        }
        break;
      }

      case 9: {
        this.setState({ opened: false, activeIndex: -1 });
        break;
      }

      case 27: {
        event.preventDefault();
        event.stopPropagation();
        this.hideDropdown();
        break;
      }

      default:
        this.setState({ activeIndex: -1 });
    }
  };

  handleShowAllLabelsAsyncToggle = () => {
    this.setState({ showAllAsync: !this.state.showAllAsync }, () => {
      this.loadOptions(true);
    });
  };

  handleOptionClick = () => {
    this.hideDropdown();
  };

  showDropdown() {
    this.setState({ opened: true, filter: this.props.value }, () => {
      this._searchbox.focus();
      this.updateDropdownHeight();
      this.loadOptions(true);
    });
  }

  hideDropdown() {
    this.setState({ opened: false, filter: '' });
  }

  // It's copy-paste from Datalist
  scrollToActiveItem() {
    if (!this._activeLi) {
      return;
    }

    const liScrollTop = this._activeLi.offsetTop - this._innerMenu.offsetTop;
    const liScrollBottom = liScrollTop + this._activeLi.clientHeight;
    const dropdownScrollTop = this._innerMenu.scrollTop;
    const dropdownHeight = this._innerMenu.clientHeight;

    let newDropdownScrollTop;

    if (liScrollTop < dropdownScrollTop) {
      newDropdownScrollTop = liScrollTop;
    } else if (liScrollBottom > dropdownScrollTop + dropdownHeight) {
      newDropdownScrollTop = liScrollBottom - dropdownHeight;
    } else {
      newDropdownScrollTop = dropdownScrollTop;
    }

    this._innerMenu.scrollTop = newDropdownScrollTop;
  }

  loadOptions(isFirstRequest = false) {
    this.props.onLoad({
      filter: isFirstRequest ? '' : this.state.filter,
      showAllLabels: this.state.showAllAsync,
    });
  }

  renderOption(option, index) {
    return (
      <li
        key={option.value}
        className={index === this.state.activeIndex ? 'active' : ''}
        ref={(el) => { if (index === this.state.activeIndex) { this._activeLi = el; } }}
        onClick={this.handleOptionClick}
      >
        <Link
          className="label-link"
          data-value={option.value}
          to={option.fullPath}
        >
          <span className="text">{option.value}</span>
        </Link>
      </li>
    );
  }

  render() {
    const {
      isAsync, name, value, loading, options, errorMessage,
    } = this.props;
    const { filter } = this.state;

    let showAllAsyncBlock = null;
    let warningBox = null;

    if (isAsync) {
      const isSpecialSelector = UserLinksBasic.isSpecialSelectorName(name);

      const isPattern = !isSpecialSelector && (filter.indexOf('*') >= 0 || filter.indexOf('?') >= 0
        || filter.indexOf('|') >= 0 || filter.indexOf('!') >= 0);

      showAllAsyncBlock = (
        <div
          role="presentation"
          className="show-all-values-btn"
          onClick={this.handleShowAllLabelsAsyncToggle}
        >
          {this.state.showAllAsync ? 'Show matched labels' : 'Show all labels'}
        </div>
      );

      if (loading && options.length === 0) {
        warningBox = (<li className="warning-box">Loading...</li>);
      } else if (errorMessage) {
        warningBox = (<li className="warning-box">{errorMessage}</li>);
      } else if (isPattern) {
        warningBox = (<li className="warning-box">Press &quot;Enter&quot; to apply pattern</li>);
      }
    }

    return (
      <div className="selector">
        <Link to={this.props.dropPath} onClick={this.props.onDropPathClick}>
          <span className="selector-drop">×</span>
        </Link>
        &nbsp;
        <span className="selector-name">{this.props.name}</span>
        <div className="label-selectpicker-placeholder">
          <div
            className={`btn-group label-selectpicker${this.state.opened ? ' open' : ''}`}
            ref={(el) => { this._selectpicker = el; }}
          >
            <button
              type="button"
              className="btn dropdown-toggle btn-default btn-xs"
              onClick={this.handleDropdownToggle}
            >
              <span className="filter-option pull-left">{value}</span>
              &nbsp;
              <span className="caret" />
            </button>
            <div className="dropdown-menu open" ref={(el) => { this._outerMenu = el; }}>
              <div className="searchbox">
                <input
                  type="text"
                  className="form-control"
                  autoComplete="off"
                  value={filter}
                  ref={(el) => { this._searchbox = el; }}
                  onChange={this.handleInputChange}
                  onKeyDown={this.handleInputKeyDown}
                />
              </div>
              <ul
                className="dropdown-menu inner"
                ref={(el) => { this._innerMenu = el; }}
              >
                {warningBox}
                {options.map((option, index) => this.renderOption(option, index))}
              </ul>
              {showAllAsyncBlock}
            </div>
          </div>
        </div>
      </div>
    );
  }
}

LabelSelector.propTypes = {
  isAsync: PropTypes.bool,
  showAllValues: PropTypes.bool.isRequired,
  name: PropTypes.string.isRequired,
  value: PropTypes.string.isRequired,
  dropPath: PropTypes.string.isRequired,
  options: PropTypes.arrayOf(LabelSelectorStateProps).isRequired,
  defaultOpened: PropTypes.bool,
  loading: PropTypes.bool,
  errorMessage: PropTypes.string,
  onSelect: PropTypes.func.isRequired,
  onLoad: PropTypes.func.isRequired,
  onDropPathClick: PropTypes.func,
};

LabelSelector.defaultProps = {
  isAsync: false,
  loading: false,
  errorMessage: '',
  defaultOpened: false,
  onDropPathClick: () => {},
};

LabelSelector.contextTypes = {
  router: PropTypes.object.isRequired,
};

export default LabelSelector;
