import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { DragLayer } from 'react-dnd';
import isEqual from 'lodash/isEqual';

import { DndDropTypes, NodeTypes } from './constants';

const collectLayer = (monitor) => ({
  item: monitor.getItem(),
  itemType: monitor.getItemType(),
  initialOffset: monitor.getInitialSourceClientOffset(),
  currentOffset: monitor.getClientOffset(),
  isDragging: monitor.isDragging(),
});

class CustomDragLayer extends PureComponent {
  getItemStyles() {
    const { initialOffset, currentOffset } = this.props;

    if (!initialOffset || !currentOffset) {
      return {
        display: '',
      };
    }

    const { x, y } = currentOffset;

    const transform = `translate(${x}px, ${y}px)`;
    return {
      transform,
      WebkitTransform: transform,
    };
  }

  renderItem() {
    const { item, itemType } = this.props;

    switch (itemType) {
      case NodeTypes.NODE: {
        const {
          hoverTitle, dropType, sourcePath, hoverPath,
        } = item;

        if (!sourcePath || !hoverPath) {
          return null;
        }

        if (isEqual(sourcePath, hoverPath)) {
          return null;
        }

        let msg;

        switch (dropType) {
          case DndDropTypes.MOVE_BEFORE:
            msg = `Move before "${hoverTitle}"`;
            break;
          case DndDropTypes.MOVE_AFTER:
            msg = `Move after "${hoverTitle}"`;
            break;
          case DndDropTypes.APPEND_HERE:
            msg = `Append to "${hoverTitle}"`;
            break;
          case DndDropTypes.PREVENT_FOR_LEAF:
            msg = 'Cannot append to leaf';
            break;
          default: {
            msg = '';
          }
        }

        if (!msg) {
          return null;
        }

        return (
          <div className="menu-editor-drag-layer__hover-info">
            {msg}
          </div>
        );
      }
      default:
        return null;
    }
  }

  render() {
    const { isDragging } = this.props;

    if (!isDragging) {
      return null;
    }

    return (
      <div className="menu-editor-drag-layer">
        <div style={this.getItemStyles()}>
          {this.renderItem()}
        </div>
      </div>
    );
  }
}

CustomDragLayer.propTypes = {
  item: PropTypes.object,
  itemType: PropTypes.string,
  initialOffset: PropTypes.shape({
    x: PropTypes.number.isRequired,
    y: PropTypes.number.isRequired,
  }),
  currentOffset: PropTypes.shape({
    x: PropTypes.number.isRequired,
    y: PropTypes.number.isRequired,
  }),
  isDragging: PropTypes.bool.isRequired,
};

CustomDragLayer.defaultProps = {
  item: {},
  itemType: '',
  initialOffset: null,
  currentOffset: null,
};

export default DragLayer(collectLayer)(CustomDragLayer);
