import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { DragDropContext } from 'react-dnd';
import HTML5Backend from 'react-dnd-html5-backend';

import MenuEditorNode from './MenuEditorTree';
import NewNodePanel from './NewNodePanel';
import CustomDragLayer from './CustomDragLayer';

import {
  changeNodeByPath, deleteNodeByPath, replaceNodeByPath,
  moveNodeAfterPath, moveNodeBeforePath, appendNodeToPath, getNodeByPath, computeAvailablePath,
} from './treeUtils';

import './style.css';

class MenuEditor extends PureComponent {
  constructor(props) {
    super(props);

    this.state = {
      treeData: props.treeData,
      editPath: [],
      focusPath: [],
    };
  }

  componentDidUpdate(prevProps) {
    if (this.props.treeData !== prevProps.treeData) {
      this.setState({ treeData: this.props.treeData, editPath: [], focusPath: [] });
    }
  }

  onEditNode = (path) => {
    this.setState({ editPath: path });
  };

  onAppendChild = (parentPath, node) => {
    let newTreeData;
    let focusPath;

    if (parentPath == null) {
      newTreeData = [...this.state.treeData];
      newTreeData.push(node);
      focusPath = [newTreeData.length - 1];
    } else {
      const parent = getNodeByPath(this.state.treeData, parentPath);
      const childrenLen = parent.children ? parent.children.length : 0;
      focusPath = [...parentPath, childrenLen];

      newTreeData = changeNodeByPath(this.state.treeData, parentPath, (parentNode) => {
        const newNode = { ...parentNode };
        const newChildren = newNode.children ? [...newNode.children] : [];
        newChildren.push(node);
        newNode.children = newChildren;
        return newNode;
      });
    }
    this.setState({ treeData: newTreeData, editPath: [], focusPath }, () => {
      this.props.onChange(newTreeData);
    });
  };

  onChangeNode = (path, node) => {
    const newTreeData = replaceNodeByPath(this.state.treeData, path, node);
    this.setState({ treeData: newTreeData, editPath: [], focusPath: path }, () => {
      this.props.onChange(newTreeData);
    });
  };

  onDeleteNode = (path) => {
    const newTreeData = deleteNodeByPath(this.state.treeData, path);
    const newFocusPath = computeAvailablePath(newTreeData, path);
    this.setState({ treeData: newTreeData, editPath: [], focusPath: newFocusPath }, () => {
      this.props.onChange(newTreeData);
    });
  };

  onToggleChildrenVisibility = (path) => {
    const newTreeData = changeNodeByPath(
      this.state.treeData,
      path,
      (node) => ({ ...node, expanded: !node.expanded }),
    );
    this.setState({ treeData: newTreeData, editPath: [], focusPath: path }, () => {
      this.props.onChange(newTreeData);
    });
  };

  onDiscardChanges = () => {
    this.setState({ editPath: [] });
  };

  onFocusNode = (path) => {
    this.setState({ focusPath: path });
  };

  onBlurNode = () => {
    this.setState({ focusPath: [] });
  };

  onBeginDrag = (path) => {
    const newTreeData = changeNodeByPath(this.state.treeData, path, (node) => {
      if (!node.expanded) {
        return node;
      }
      return { ...node, expanded: false };
    });
    this.setState({ treeData: newTreeData, editPath: [], focusPath: path }, () => {
      this.props.onChange(newTreeData);
    });
  };

  onMoveNode = (type, sourcePath, targetPath) => {
    let result;

    switch (type) {
      case 'before': {
        result = moveNodeBeforePath(this.state.treeData, sourcePath, targetPath);
        break;
      }
      case 'after': {
        result = moveNodeAfterPath(this.state.treeData, sourcePath, targetPath);
        break;
      }
      case 'here': {
        result = appendNodeToPath(this.state.treeData, sourcePath, targetPath);
        break;
      }
      default: {
        return;
      }
    }

    this.setState({ editPath: [], treeData: result.treeData, focusPath: result.newPath }, () => {
      this.props.onChange(result.treeData);
    });
  };

  renderNode(index, node) {
    return (
      <MenuEditorNode
        key={`menu-editor-${index}`}
        path={[index]}
        node={node}
        editPath={this.state.editPath}
        focusPath={this.state.focusPath}
        onEditNode={this.onEditNode}
        onDeleteNode={this.onDeleteNode}
        onToggleChildrenVisibility={this.onToggleChildrenVisibility}
        onChangeNode={this.onChangeNode}
        onAppendChild={this.onAppendChild}
        onDiscardChanges={this.onDiscardChanges}
        onFocus={this.onFocusNode}
        onBlur={this.onBlurNode}
        onBeginDrag={this.onBeginDrag}
        onMoveNode={this.onMoveNode}
      />
    );
  }

  render() {
    const { treeData, editPath } = this.state;

    const nodeElements = treeData.map((node, index) => this.renderNode(index, node));

    const newNodePanel = (
      <NewNodePanel
        editPath={editPath}
        onEdit={this.onEditNode}
        onAppendChild={this.onAppendChild}
        onDiscard={this.onDiscardChanges}
      />
    );

    return (
      <div className="menu-editor">
        {nodeElements}
        {newNodePanel}
        <CustomDragLayer />
      </div>
    );
  }
}

MenuEditor.propTypes = {
  treeData: PropTypes.arrayOf(PropTypes.object),
  onChange: PropTypes.func,
};

MenuEditor.defaultProps = {
  treeData: [],
  onChange: () => {},
};

export default DragDropContext(HTML5Backend)(MenuEditor);
