import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEqual from 'lodash/isEqual';

import MenuReadonlyNode from './MenuReadonlyNode';
import MenuEditableNode from './MenuEditableNode';
import NewNodePanel from './NewNodePanel';

class MenuEditorTree extends PureComponent {
  renderReadonlyNode(path, node) {
    return (
      <MenuReadonlyNode
        path={path}
        data={node}
        focused={isEqual(this.props.focusPath, this.props.path)}
        onEdit={this.props.onEditNode}
        onDelete={this.props.onDeleteNode}
        onToggle={this.props.onToggleChildrenVisibility}
        onFocus={this.props.onFocus}
        onBlur={this.props.onBlur}
        onBeginDrag={this.props.onBeginDrag}
        onMoveNode={this.props.onMoveNode}
      />
    );
  }

  renderEditableNode(path, node) {
    return (
      <MenuEditableNode
        path={path}
        data={node}
        onSubmit={this.props.onChangeNode}
        onDiscard={this.props.onDiscardChanges}
      />
    );
  }

  renderChild(index, child) {
    return (
      <MenuEditorTree
        // eslint-disable-next-line react/no-array-index-key
        key={index}
        path={[...this.props.path, index]}
        editPath={this.props.editPath}
        focusPath={this.props.focusPath}
        node={child}
        onEditNode={this.props.onEditNode}
        onDeleteNode={this.props.onDeleteNode}
        onToggleChildrenVisibility={this.props.onToggleChildrenVisibility}
        onChangeNode={this.props.onChangeNode}
        onAppendChild={this.props.onAppendChild}
        onDiscardChanges={this.props.onDiscardChanges}
        onFocus={this.props.onFocus}
        onBlur={this.props.onBlur}
        onBeginDrag={this.props.onBeginDrag}
        onMoveNode={this.props.onMoveNode}
      />
    );
  }

  renderNewNodePanel(editPath, parentPath) {
    return (
      <NewNodePanel
        isSmall
        editPath={editPath}
        parentPath={parentPath}
        onEdit={this.props.onEditNode}
        onAppendChild={this.props.onAppendChild}
        onDiscard={this.props.onDiscardChanges}
      />
    );
  }

  render() {
    const { path, editPath, node } = this.props;

    const isTree = !node.url;

    let nodeElement;

    if (isEqual(path, editPath)) {
      nodeElement = this.renderEditableNode(path, node);
    } else {
      nodeElement = this.renderReadonlyNode(path, node);
    }

    let childrenElement = null;

    if (isTree && node.expanded) {
      const children = node.children || [];

      const childrenElements = children.map((child, index) => this.renderChild(index, child));
      const newNodePanel = this.renderNewNodePanel(editPath, path);

      childrenElement = (
        <ul className="menu-editor-tree__children">
          {childrenElements}
          {newNodePanel}
        </ul>
      );
    }

    return (
      <li className="menu-editor-tree">
        {nodeElement}
        {childrenElement}
      </li>
    );
  }
}

MenuEditorTree.propTypes = {
  path: PropTypes.arrayOf(PropTypes.number).isRequired,
  editPath: PropTypes.arrayOf(PropTypes.number).isRequired,
  focusPath: PropTypes.arrayOf(PropTypes.number).isRequired,
  node: PropTypes.object.isRequired,
  onEditNode: PropTypes.func.isRequired,
  onDeleteNode: PropTypes.func.isRequired,
  onToggleChildrenVisibility: PropTypes.func.isRequired,
  onChangeNode: PropTypes.func.isRequired,
  onAppendChild: PropTypes.func.isRequired,
  onDiscardChanges: PropTypes.func.isRequired,
  onFocus: PropTypes.func.isRequired,
  onBlur: PropTypes.func.isRequired,
  onBeginDrag: PropTypes.func.isRequired,
  onMoveNode: PropTypes.func.isRequired,
};

export default MenuEditorTree;
