import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { findDOMNode } from 'react-dom';
import { DragSource, DropTarget } from 'react-dnd';
import { getEmptyImage } from 'react-dnd-html5-backend';
import isEqual from 'lodash/isEqual';

import { DndDropTypes, NodeTypes } from './constants';

import './style.css';

const NODE_EDGE_HEIGHT_PX = 15;

const menuNodeSource = {
  beginDrag(props) {
    props.onBeginDrag(props.path);
    return {
      sourcePath: props.path,
      sourceTitle: props.data.title,
      hoverPath: null,
      dropType: null,
    };
  },
};

const collectSource = (connect, monitor) => ({
  connectDragSource: connect.dragSource(),
  connectDragPreview: connect.dragPreview(),
  isDragging: monitor.isDragging(),
});

const menuNodeTarget = {
  hover(props, monitor, component) {
    const item = monitor.getItem();

    const { sourcePath } = item;
    const targetPath = props.path;

    item.hoverPath = targetPath;
    item.hoverTitle = props.data.title;

    let dropType;

    if (isEqual(sourcePath, targetPath)) {
      dropType = DndDropTypes.PREVENT_FOR_SOURCE;
    } else {
      const targetIsTree = !props.data.url;

      // eslint-disable-next-line react/no-find-dom-node
      const hoverBoundingRect = findDOMNode(component).getBoundingClientRect();

      const clientOffset = monitor.getClientOffset();

      const hoverClientY = clientOffset.y - hoverBoundingRect.top;

      const clientHeight = hoverBoundingRect.height;

      if (hoverClientY < NODE_EDGE_HEIGHT_PX) {
        dropType = DndDropTypes.MOVE_BEFORE;
      } else if (hoverClientY > clientHeight - NODE_EDGE_HEIGHT_PX) {
        dropType = DndDropTypes.MOVE_AFTER;
      } else if (targetIsTree) {
        dropType = DndDropTypes.APPEND_HERE;
      } else {
        dropType = DndDropTypes.PREVENT_FOR_LEAF;
      }
    }

    item.dropType = dropType;
    component.setState({ dropType });

    return {
      ...item,
      dropType,
      hoverPath: targetPath,
      hoverTitle: props.data.title,
    };
  },
  drop(props, monitor) {
    const item = monitor.getItem();
    const { sourcePath } = item;
    const targetPath = props.path;

    if (isEqual(sourcePath, targetPath)) {
      return;
    }

    props.onMoveNode(item.dropType, sourcePath, targetPath);
  },
};

const collectTarget = (connect, monitor) => ({
  connectDropTarget: connect.dropTarget(),
  isOver: monitor.isOver(),
  hoverPath: monitor.getItem() ? monitor.getItem().hoverPath : '',
  dropType: monitor.getItem() ? monitor.getItem().dropType : '',
});

class MenuReadonlyNode extends PureComponent {
  componentDidMount() {
    this.props.connectDragPreview(getEmptyImage(), { captureDraggingState: true });
    if (this.props.focused) {
      this.rootEl.focus();
    }
  }

  componentDidUpdate() {
    if (this.props.focused) {
      this.rootEl.focus();
    }
  }

  onToggleClick = (event) => {
    event.preventDefault();
    this.props.onToggle(this.props.path);
  };

  onEditClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.props.onEdit(this.props.path);
  };

  onDeleteClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.props.onDelete(this.props.path);
  };

  onFocus = (event) => {
    event.preventDefault();
    this.props.onFocus(this.props.path);
  };

  onBlur = (event) => {
    event.preventDefault();
    this.props.onBlur();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      // Edit by "Enter" press
      event.preventDefault();
      this.props.onEdit(this.props.path);
    } else if (event.which === 27) {
      // Blur by "Esc" press
      event.preventDefault();
      this.props.onBlur();
    } else if (event.which === 37) {
      // Collapse node (or parent in case of collapsed node) by left arrow press
      if (this.props.data.expanded) {
        event.preventDefault();
        this.props.onToggle(this.props.path);
      } else if (this.props.path.length > 1) {
        event.preventDefault();
        this.props.onToggle(this.props.path.slice(0, this.props.path.length - 1));
      }
    } else if (event.which === 39 && !this.props.data.expanded) {
      // Expand node by right arrow press
      event.preventDefault();
      this.props.onToggle(this.props.path);
    } else if (event.which === 46) {
      // Delete node by "Delete" press
      event.preventDefault();
      this.props.onDelete(this.props.path);
    }
  };

  render() {
    const {
      data, focused,
      connectDropTarget, connectDragSource,
      isDragging, isOver, dropType,
    } = this.props;
    const { expanded } = data;

    let hoverModeClassName = '';

    if (!isDragging && isOver) {
      switch (dropType) {
        case DndDropTypes.MOVE_BEFORE:
          hoverModeClassName = ' menu-editor-node_move-before';
          break;
        case DndDropTypes.MOVE_AFTER:
          hoverModeClassName = ' menu-editor-node_move-after';
          break;
        case DndDropTypes.APPEND_HERE:
          hoverModeClassName = ' menu-editor-node_append-here';
          break;
        case DndDropTypes.PREVENT_FOR_LEAF:
          hoverModeClassName = ' menu-editor-node_prevent';
          break;
        default: {
          hoverModeClassName = '';
        }
      }
    }

    // eslint-disable-next-line no-mixed-operators
    const isTree = data.children && data.children.length > 0 || !data.url;

    const toggler = (
      <div className="menu-editor-node__toggler">
        {isTree && (
          <i className={`glyphicon ${expanded ? 'glyphicon-menu-down' : 'glyphicon-menu-right'}`} />)}
      </div>
    );

    const content = (
      <div className="menu-editor-node__content">
        <div className="menu-editor-node__field menu-editor-node__title">
          {data.title}
        </div>
        <div className="menu-editor-node__field menu-editor-node__url">
          {data.url}
        </div>
        <div className="menu-editor-node__field menu-editor-node__selectors">
          {data.selectors}
        </div>
      </div>
    );

    const buttons = (
      <div className="menu-editor-node__buttons">
        <i
          className="glyphicon glyphicon-pencil"
          role="link" tabIndex={-1}
          onClick={this.onEditClick}
        />
        <i
          className="glyphicon glyphicon-trash"
          role="link" tabIndex={-1}
          onClick={this.onDeleteClick}
        />
      </div>
    );

    return connectDropTarget(connectDragSource(
      <div
        tabIndex={0}
        role="link"
        className={`menu-editor-node menu-editor-node_readonly${hoverModeClassName}${focused ? ' menu-editor-node_active' : ''}`}
        onClick={this.onToggleClick}
        onDoubleClick={isTree ? null : this.onEditClick}
        onFocus={this.onFocus}
        onBlur={this.onBlur}
        onKeyDown={this.onKeyDown}
        ref={(el) => { this.rootEl = el; }}
      >
        {toggler}
        {content}
        {buttons}
      </div>,
    ));
  }
}

MenuReadonlyNode.propTypes = {
  path: PropTypes.array.isRequired,
  data: PropTypes.object.isRequired,
  focused: PropTypes.bool.isRequired,
  onEdit: PropTypes.func.isRequired,
  onDelete: PropTypes.func.isRequired,
  onToggle: PropTypes.func.isRequired,
  onFocus: PropTypes.func.isRequired,
  onBlur: PropTypes.func.isRequired,
  // DnD callbacks
  /* eslint-disable react/no-unused-prop-types */
  onBeginDrag: PropTypes.func.isRequired,
  onMoveNode: PropTypes.func.isRequired,
  // DnD functions
  connectDropTarget: PropTypes.func.isRequired,
  connectDragSource: PropTypes.func.isRequired,
  connectDragPreview: PropTypes.func.isRequired,
  // DnD properties
  isDragging: PropTypes.bool.isRequired,
  isOver: PropTypes.bool.isRequired,
  dropType: PropTypes.string,
};

MenuReadonlyNode.defaultProps = {
  dropType: null,
};

export default
DropTarget(NodeTypes.NODE, menuNodeTarget, collectTarget)(
  DragSource(NodeTypes.NODE, menuNodeSource, collectSource)(MenuReadonlyNode),
);
