/**
 * Get node by path from tree data
 *
 * @param treeData - array of tree data nodes
 * @param path - array of number indexes
 * @returns node required by path
 */
export function getNodeByPath(treeData, path) {
  if (path.length === 0) {
    throw new Error('path cannot be empty');
  }

  const index = path[0];

  if (path.length === 1) {
    return treeData[path[0]];
  }

  const currentNode = treeData[index];

  if (!currentNode.children) {
    throw new Error('unknown children');
  }

  const nextPath = path.slice(1);

  return getNodeByPath(currentNode.children, nextPath);
}

/**
 * Insert node to required place by path in treeData
 * Last place index can be greater than parent children index (in that case we insert at last).
 * See https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Array/splice
 *
 * @param treeData - array of tree nodes
 * @param path - array of number indexes
 * @param node - new node
 * @returns new tree data after node inserting
 */
export function insertNodeByPath(treeData, path, node) {
  if (path.length === 0) {
    throw new Error('path cannot be empty');
  }

  const index = path[0];

  if (path.length === 1) {
    const newTreeData = [...treeData];
    newTreeData.splice(index, 0, node);
    return newTreeData;
  }

  const currentNode = treeData[index];

  if (!currentNode.children) {
    throw new Error('unknown children');
  }

  const nextPath = path.slice(1);

  const newChildren = insertNodeByPath(currentNode.children, nextPath, node);

  const newCurrentNode = {
    ...currentNode,
    children: newChildren,
  };

  const newTreeData = [...treeData];
  newTreeData[index] = newCurrentNode;

  return newTreeData;
}

/**
 * Replace node to required place by path in treeData
 * Last place index can be greater than parent children index (in that case we insert at last).
 *
 * @param treeData - array of tree nodes
 * @param path - array of number indexes
 * @param node - new node
 * @returns new tree data after node replacement
 */
export function replaceNodeByPath(treeData, path, node) {
  if (path.length === 0) {
    throw new Error('path cannot be empty');
  }

  const index = path[0];

  if (path.length === 1) {
    const newTreeData = [...treeData];
    newTreeData.splice(index, 1, node);
    return newTreeData;
  }

  const currentNode = treeData[index];

  if (!currentNode.children) {
    throw new Error('unknown children');
  }

  const nextPath = path.slice(1);

  const newChildren = replaceNodeByPath(currentNode.children, nextPath, node);

  const newCurrentNode = {
    ...currentNode,
    children: newChildren,
  };

  const newTreeData = [...treeData];
  newTreeData[index] = newCurrentNode;

  return newTreeData;
}

/**
 * Delete node from required place by path in treeData
 *
 * @param treeData - array of tree nodes
 * @param path - array of number indexes
 * @returns new tree data after node deletion
 */
export function deleteNodeByPath(treeData, path) {
  if (path.length === 0) {
    throw new Error('path cannot be empty');
  }

  const index = path[0];

  if (path.length === 1) {
    const newTreeData = [...treeData];
    newTreeData.splice(index, 1);
    return newTreeData;
  }

  const currentNode = treeData[index];

  if (!currentNode.children) {
    throw new Error('unknown children');
  }

  const nextPath = path.slice(1);

  const newChildren = deleteNodeByPath(currentNode.children, nextPath);

  const newCurrentNode = {
    ...currentNode,
    children: newChildren,
  };

  const newTreeData = [...treeData];
  newTreeData[index] = newCurrentNode;

  return newTreeData;
}

/**
 * Change node in required place by path in treeData
 *
 * @param treeData - array of tree nodes
 * @param path - array of number indexes
 * @param nodeFunc - function to change old node by path to new node
 * @returns new tree data after node change
 */
export function changeNodeByPath(treeData, path, nodeFunc) {
  if (path.length === 0) {
    throw new Error('path cannot be empty');
  }

  const index = path[0];

  if (path.length === 1) {
    const newTreeData = [...treeData];
    const oldNode = newTreeData[index];
    const newNode = nodeFunc(oldNode);
    newTreeData.splice(index, 1, newNode);
    return newTreeData;
  }

  const currentNode = treeData[index];

  if (!currentNode.children) {
    throw new Error('unknown children');
  }

  const nextPath = path.slice(1);

  const newChildren = changeNodeByPath(currentNode.children, nextPath, nodeFunc);

  const newCurrentNode = {
    ...currentNode,
    children: newChildren,
  };

  const newTreeData = [...treeData];
  newTreeData[index] = newCurrentNode;

  return newTreeData;
}

/**
 * Lexicographical comparison of node pathes
 *
 * @param path1 - array of number indexes for first node
 * @param path2 - array of number indexes for second node
 * @returns -1 in case of path1 < path2, 1 is case of path1 > path2, 0 otherwise
 */
export function comparePathes(path1, path2) {
  for (let i = 0; i < path1.length; ++i) {
    if (i >= path2.length) {
      return 1;
    }

    const index1 = path1[i];
    const index2 = path2[i];

    if (index1 !== index2) {
      return index1 < index2 ? -1 : 1;
    }
  }
  return path1.length === path2.length ? 0 : -1;
}

/**
 * Compute common parent path length for two paths
 * Use path1.slice(0, parentLen) to get common parent of path1 and path2.
 *
 * @param path1
 * @param path2
 * @returns length of common parent for path1 and path2.
 */
export function computeCommonParentPathLen(path1, path2) {
  for (let i = 0; i < path1.length; ++i) {
    if (i >= path2.length) {
      return path2.length;
    }

    const index1 = path1[i];
    const index2 = path2[i];

    if (index1 !== index2) {
      return i;
    }
  }
  return path1.length;
}

/**
 * Compute new node path after moving using sourcePath and targetPath values
 * Need for moveNode... functions
 *
 * @param sourcePath
 * @param targetPath
 * @returns new node path after moving
 */
export function computeNewPathAfterMoving(sourcePath, targetPath) {
  if (sourcePath.length === 0 || targetPath.length === 0) {
    return [];
  }

  let newPath;

  const comp = comparePathes(sourcePath, targetPath);

  if (comp === 0) {
    newPath = sourcePath;
  } else if (comp < 0) {
    const parentPathLen = computeCommonParentPathLen(sourcePath, targetPath);

    if (parentPathLen === sourcePath.length - 1) {
      // Node path has shifted if we have deleted source node
      newPath = [...targetPath];
      // eslint-disable-next-line no-plusplus
      newPath[parentPathLen]--;
    } else {
      newPath = targetPath;
    }
  } else {
    newPath = targetPath;
  }

  return newPath;
}

/**
 * Move node from sourcePath to node before targetPath (computed for start state) in treeData
 *
 * @param treeData
 * @param sourcePath
 * @param targetPath
 * @returns result tree data after moving and new path for moved node
 */
export function moveNodeBeforePath(treeData, sourcePath, targetPath) {
  const sourceNode = getNodeByPath(treeData, sourcePath);

  const newPath = computeNewPathAfterMoving(sourcePath, targetPath);

  if (newPath === sourcePath) {
    return { treeData, newPath: sourcePath };
  }

  const newTreeData = insertNodeByPath(deleteNodeByPath(treeData, sourcePath), newPath, sourceNode);

  return { treeData: newTreeData, newPath };
}

/**
 * Move node from sourcePath to node after targetPath (computed for start state) in treeData
 *
 * @param treeData
 * @param sourcePath
 * @param targetPath
 * @returns result tree data after moving and new path for moved node
 */
export function moveNodeAfterPath(treeData, sourcePath, targetPath) {
  const newTargetPath = [...targetPath];
  // eslint-disable-next-line no-plusplus
  newTargetPath[newTargetPath.length - 1]++;

  return moveNodeBeforePath(treeData, sourcePath, newTargetPath);
}

/**
 * Move node from sourcePath to node children by targetPath (computed for start state) in treeData
 *
 * @param treeData
 * @param sourcePath
 * @param targetPath
 * @returns result tree data after moving and new path for moved node
 */
export function appendNodeToPath(treeData, sourcePath, targetPath) {
  const sourceNode = getNodeByPath(treeData, sourcePath);

  const changedTargetPath = computeNewPathAfterMoving(sourcePath, targetPath);

  if (changedTargetPath === sourcePath) {
    return { treeData, newPath: sourcePath };
  }

  let newTreeData = changeNodeByPath(treeData, targetPath, (node) => {
    const children = node.children ? [...node.children] : [];
    children.push(sourceNode);
    return { ...node, children, expanded: true };
  });

  newTreeData = deleteNodeByPath(newTreeData, sourcePath);

  const newNodeIndex = getNodeByPath(newTreeData, changedTargetPath).children.length - 1;
  const newPath = [...changedTargetPath, newNodeIndex];

  return { treeData: newTreeData, newPath };
}

/**
 * Compute available path for tree data (need to set focus after deletion):
 * - returns this path if node exists by path
 * - otherwise find last child in parent
 * - otherwise find first available parent
 * @param treeData
 * @param path
 * @returns {*} First available node path
 */
export function computeAvailablePath(treeData, path) {
  if (treeData.length === 0 || path.length === 0) {
    return [];
  }

  if (path[0] >= treeData.length) {
    return [treeData.length - 1];
  }

  return [path[0], ...computeAvailablePath(treeData[path[0]].children || [], path.slice(1))];
}
