/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import TableEditor from '../TableEditor';
import TableViewer from '../TableViewer';

import wrapHandler from '../../utils/handlerWrapper';

const SELECTOR_SHAPE = PropTypes.shape({
  name: PropTypes.string,
  value: PropTypes.string,
});

function mapToList(map = {}) {
  const result = [];
  // eslint-disable-next-line guard-for-in,no-restricted-syntax
  for (const name in map) {
    result.push({ name, value: map[name] });
  }
  return result;
}

function listToMap(list = []) {
  const map = {};
  for (let i = 0; i < list.length; ++i) {
    const entry = list[i];
    map[entry.name] = entry.value;
  }
  return map;
}

class ReadonlyRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td>{value.name}</td>
        <td>{value.value}</td>
        {this.onDelete
        && (
        <td className="actions">
          <button type="button" onClick={this.onDelete}>
            <i className="glyphicon glyphicon-remove" />
          </button>
        </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: SELECTOR_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { ...props.value };
  }

  componentDidMount() {
    this.nameInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onInputChange = (event) => {
    const { name, value } = event.target;
    this.setState({ [name]: value });
  };

  doComplete() {
    const { onOk, index } = this.props;
    const { name, value } = this.state;
    onOk(index, { name, value });
  }

  render() {
    const { index } = this.props;
    const { name, value } = this.state;
    return (
      <tr onKeyDown={this.onKeyDown}>
        <td>{index >= 0 ? index + 1 : ''}</td>
        <td>
          <input
            type="text" name="name" className="form-control"
            value={name}
            onChange={this.onInputChange}
            ref={(input) => { this.nameInput = input; }}
          />
        </td>
        <td>
          <input
            type="text" name="value" className="form-control"
            value={value}
            onChange={this.onInputChange}
          />
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: SELECTOR_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: { name: '', value: '' },
};

export const ReadonlyTable = ({ map, limit }) => (
  <TableViewer
    columns={['#', 'Name', 'Value']}
    values={mapToList(map)}
    row={ReadonlyRow}
    limit={limit}
  />
);

ReadonlyTable.propTypes = {
  map: PropTypes.object,
  limit: PropTypes.number.isRequired,
};

ReadonlyTable.defaultProps = {
  map: {},
};

export const EditableTable = ({ map = {}, onUpdate }) => {
  const list = mapToList(map);

  const onDeleteClick = (index) => {
    const newList = [...list];
    newList.splice(index, 1);
    const newMap = listToMap(newList);
    onUpdate(newMap);
  };

  const onUpdateClick = (index, value) => {
    const newList = [...list];
    if (index >= 0) {
      newList[index] = value;
    } else {
      newList.push(value);
    }
    const newMap = listToMap(newList);
    onUpdate(newMap);
  };

  return (
    <TableEditor
      columns={['#', 'Name', 'Value']}
      values={list}
      readonlyRow={ReadonlyRow}
      editableRow={EditableRow}
      onDelete={onDeleteClick}
      onUpdate={onUpdateClick}
    />
  );
};

EditableTable.propTypes = {
  map: PropTypes.object,
  onUpdate: PropTypes.func.isRequired,
};

EditableTable.defaultProps = {
  map: {},
};
