/* eslint-disable max-classes-per-file */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import wrapHandler from '../../utils/handlerWrapper';
import TableEditor from '../TableEditor';
import TableViewer from '../TableViewer';

const PARAMETER_SHAPE = PropTypes.shape({
  name: PropTypes.string,
  value: PropTypes.string,
});

const PARAMETER_DEFAULT_VALUE = {
  name: '',
  value: '',
};

class ReadonlyRow extends PureComponent {
  constructor(props) {
    super(props);
    this.onClick = wrapHandler(props.index, props.onClick);
    this.onDelete = wrapHandler(props.index, props.onDelete);
  }

  render() {
    const { index, value } = this.props;
    return (
      <tr onClick={this.onClick}>
        <td>{index + 1}</td>
        <td>{value.name}</td>
        <td>{value.value}</td>
        {this.onDelete && (
          <td className="actions">
            <button type="button" onClick={this.onDelete}>
              <i className="glyphicon glyphicon-remove" />
            </button>
          </td>
        )}
      </tr>
    );
  }
}

ReadonlyRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PARAMETER_SHAPE.isRequired,
  onClick: PropTypes.func,
  onDelete: PropTypes.func,
};

ReadonlyRow.defaultProps = {
  onClick: null,
  onDelete: null,
};

class EditableRow extends PureComponent {
  constructor(props) {
    super(props);
    this.state = this.props.value;
  }

  componentDidMount() {
    this.nameInput.focus();
  }

  onOkClick = (event) => {
    event.preventDefault();
    event.stopPropagation();
    this.doComplete();
  };

  onKeyDown = (event) => {
    if (event.which === 13) {
      event.preventDefault();
      event.stopPropagation();
      this.doComplete();
    } else if (event.which === 27) {
      event.preventDefault();
      this.props.onCancel();
    }
  };

  onNameInputChange = (event) => {
    this.setState({ name: event.target.value });
  };

  onValueInputChange = (event) => {
    this.setState({ value: event.target.value });
  };

  doComplete() {
    const { onOk, index } = this.props;
    onOk(index, this.state);
  }

  render() {
    const { index } = this.props;
    const { name, value } = this.state;
    return (
      <tr onKeyDown={this.onKeyDown}>
        <td>{index >= 0 ? index + 1 : ''}</td>
        <td>
          <input
            type="text" className="form-control"
            value={name} onChange={this.onNameInputChange}
            ref={(input) => { this.nameInput = input; }}
          />
        </td>
        <td>
          <input
            type="text" className="form-control"
            value={value} onChange={this.onValueInputChange}
          />
        </td>
        <td className="actions">
          <button type="button" onClick={this.onOkClick}>
            <i className="glyphicon glyphicon-ok" />
          </button>
        </td>
      </tr>
    );
  }
}

EditableRow.propTypes = {
  index: PropTypes.number.isRequired,
  value: PARAMETER_SHAPE,
  onOk: PropTypes.func.isRequired,
  onCancel: PropTypes.func.isRequired,
};

EditableRow.defaultProps = {
  value: PARAMETER_DEFAULT_VALUE,
};

export const ReadonlyTable = ({ parameters }) => (
  <TableViewer
    columns={['#', 'Name', 'Value']}
    row={ReadonlyRow}
    values={parameters}
    limit={5}
  />
);

ReadonlyTable.propTypes = {
  parameters: PropTypes.arrayOf(PARAMETER_SHAPE),
};

ReadonlyTable.defaultProps = {
  parameters: [],
};

export const EditableTable = ({ parameters, onDelete, onUpdate }) => (
  <TableEditor
    columns={['#', 'Name', 'Value']}
    values={parameters}
    readonlyRow={ReadonlyRow}
    editableRow={EditableRow}
    onDelete={onDelete}
    onUpdate={onUpdate}
  />
);

EditableTable.propTypes = {
  parameters: PropTypes.arrayOf(PARAMETER_SHAPE),
  onDelete: PropTypes.func.isRequired,
  onUpdate: PropTypes.func.isRequired,
};

EditableTable.defaultProps = {
  parameters: [],
};
