/* eslint-disable no-param-reassign */

const INDENT_PATTERN = /^\s*/;
const TAB_TEXT = '  ';

/* Functions to edit text area and set cursor position  */
function setText(event, $textArea, newText, onChange) {
  event.preventDefault();
  $textArea.value = newText;
  onChange(newText);
}

function insertText(event, $textArea, text, insText, pos, onChange) {
  const newText = text.substring(0, pos) + insText + text.substring(pos);
  setText(event, $textArea, newText, onChange);
}

function setPosition($textArea, pos) {
  $textArea.setSelectionRange(pos, pos);
}

function insertTextAndSetPosition(event, $textArea, text, insText, pos, onChange) {
  insertText(event, $textArea, text, insText, pos, onChange);
  setPosition($textArea, pos + insText.length);
}

function removeText(event, $textArea, text, start, end, onChange) {
  const newText = text.substring(0, start) + text.substring(end);
  setText(event, $textArea, newText, onChange);
  setPosition($textArea, start);
}

/* Other utilities functions */
function getFirstLineBeforePosition(text, pos) {
  const startText = text.substring(0, pos);

  let line;

  const i = startText.lastIndexOf('\n');
  if (i >= 0) {
    line = startText.substring(i + 1);
  } else {
    line = startText;
  }
  return line;
}

function getCurrentIndent(text, pos) {
  const line = getFirstLineBeforePosition(text, pos);
  const result = INDENT_PATTERN.exec(line);
  return result ? result[0] : '';
}

function isBracket(open) {
  return open === '(' || open === '{';
}

/**
 * Check key press to support key behavior for pair symbols:
 * create, close or remove pair and correct indent inside pairs
 *
 * @param {KeyboardEvent} event
 * @param {HTMLTextAreaElement} $textArea
 * @param {String} text
 * @param {number} pos
 * @param {String} open
 * @param {String} close
 * @param {function} onChange
 */
function checkKeyPressForPairs(event, $textArea, text, pos, open, close, onChange) {
  if (event.key.length === 1) {
    const key = event.key[0];

    if (key === close && key === text[pos]) {
      // Bypass close symbol
      insertText(event, $textArea, text, '', pos, onChange);
      setPosition($textArea, pos + 1);
    } else if (key === open && (pos === 0 || text[pos - 1] !== open)) {
      // Make new pair
      insertText(event, $textArea, text, `${open}${close}`, pos, onChange);
      setPosition($textArea, pos + 1);
    }
  }

  // Actions inside existing pair
  if (pos > 0 && text[pos - 1] === open && text[pos] === close) {
    if (event.key === 'Backspace') {
      removeText(event, $textArea, text, pos - 1, pos + 1, onChange);
    } else if (event.key === 'Enter') {
      const indentInNewLine = `\n${getCurrentIndent(text, pos)}`;
      const insertedText = indentInNewLine + TAB_TEXT + indentInNewLine;
      const newPos = pos + indentInNewLine.length + TAB_TEXT.length;
      insertText(event, $textArea, text, insertedText, pos, onChange);
      setPosition($textArea, newPos);
    } else if (event.key === open && isBracket(open)) {
      insertText(event, $textArea, text, `${open}${close}`, pos, onChange);
      setPosition($textArea, pos + 1);
    }
  }
}

/**
 * Handle text area keydown to support automatic bracket creation/removing and formatting
 *
 * @param {KeyboardEvent} event
 * @param {HTMLTextAreaElement} $textArea
 * @param {function} onChange
 */
function handleTextEditorKeyDown(event, $textArea, onChange) {
  let pos;
  {
    const selStart = $textArea.selectionStart;
    const selEnd = $textArea.selectionEnd;
    if (selStart !== selEnd) {
      return;
    }
    pos = selStart;
  }

  const text = $textArea.value;

  switch (event.key) {
    case 'Tab': {
      insertTextAndSetPosition(event, $textArea, text, TAB_TEXT, pos, onChange);
      break;
    }
    case 'Enter': {
      const indentInNewLine = `\n${getCurrentIndent(text, pos)}`;
      insertTextAndSetPosition(event, $textArea, text, indentInNewLine, pos, onChange);
      break;
    }
    default:
      break;
  }

  checkKeyPressForPairs(event, $textArea, text, pos, '"', '"', onChange);
  checkKeyPressForPairs(event, $textArea, text, pos, '(', ')', onChange);
  checkKeyPressForPairs(event, $textArea, text, pos, '{', '}', onChange);
}

export default handleTextEditorKeyDown;
