/* eslint-disable jsx-a11y/interactive-supports-focus */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import Tooltip from 'react-bootstrap/es/Tooltip';
import Overlay from 'react-bootstrap/es/Overlay';

import { copyContent } from '../../utils/clipboard';
import createShortUrl from '../../utils/createShortUrl';
import ShortLinkModal from './ShortLinkModal';

function allowToCopyText() {
  return new Promise((resolve) => {
    try {
      navigator.permissions.query({ name: 'clipboard-write' }).then((result) => {
        if (result.state === 'granted' || result.state === 'prompt') {
          resolve(true);
        } else {
          console.error(`failed to allow clipboard-write permission, status: ${result.state}`);
          resolve(false);
        }
      })
        .catch((e) => {
          console.error(`failed to allow clipboard-write permission, error: ${e}`);
          resolve(false);
        });
    } catch (e) {
      console.error('clipboard-write permission is not supported');
      resolve(false);
    }
  });
}

class QuickCopyButton extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { showTooltip: false, showModal: false, shortUrl: '' };
  }

  onClick = () => {
    createShortUrl(this.props.url).then((shortUrl) => allowToCopyText().then((result) => {
      if (result) {
        copyContent(shortUrl);
        this.setState({ showTooltip: true }, () => {
          window.setTimeout(() => {
            this.setState({ showTooltip: false });
          }, 1000);
        });
      } else {
        this.setState({ showModal: true, shortUrl });
      }
    }));
  };

  onHideModal = () => {
    this.setState({ showModal: false, shortUrl: '' });
  };

  render() {
    const tooltip = (
      <Tooltip id="copy-button-tooltip">
        Copied!
      </Tooltip>
    );

    const button = (
      <div className="btn-group" style={{ marginLeft: '-10px' }}>
        <button
          type="button"
          className="btn"
          style={{ background: 'transparent', boxShadow: 'none' }}
          onClick={this.onClick}
          ref={(el) => { this._button = el; }}
        >
          <i className="glyphicon glyphicon-link" />
        </button>
      </div>
    );

    return (
      <>
        {button}
        <Overlay
          target={this._button}
          show={this.state.showTooltip}
          placement="top"
        >
          {tooltip}
        </Overlay>
        <ShortLinkModal
          link={this.state.shortUrl}
          isOpen={this.state.showModal}
          onCancel={this.onHideModal}
        />
      </>
    );
  }
}

QuickCopyButton.propTypes = {
  url: PropTypes.string.isRequired,
};

export default QuickCopyButton;
