import React, { Component } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';

import SidebarMenu from '../SidebarMenu';
import { clearProjectAlertStats, loadProjectAlertStats } from '../../../store/reducers/alerts/projectAlertStats';
import ProjectAlertStats from './ProjectAlertStats';

class AlertSidebarMenu extends Component {
  componentDidMount() {
    this.runCoro(this.props.projectId);
  }

  componentDidUpdate(prevProps) {
    if (this.props.projectId !== prevProps.projectId) {
      this.clearCoro();
      this.runCoro(this.props.projectId);
    }
  }

  componentWillUnmount() {
    this.props.clearProjectAlertStats();
    this.clearCoro();
  }

  runCoro = (projectId) => {
    try {
      this.props.loadProjectAlertStats(projectId);
    } catch (e) {
      console.error('failed to load project alert stats', e);
    }
    this._coroHandler = window.setTimeout(() => {
      this.runCoro(projectId);
    }, 60000);
  };

  clearCoro = () => {
    if (this._coroHandler != null) {
      window.clearTimeout(this._coroHandler);
    }
  };

  toggleChildren = (event) => {
    event.preventDefault();
    const { onOpenToggle } = this.props;
    if (onOpenToggle) {
      onOpenToggle();
    }
  };

  render() {
    const {
      projectId, subMenus, opened, projectAlertStats,
    } = this.props;

    let childrenContainer;
    if (subMenus.length > 0) {
      const childrenElements = subMenus
        .map((child) => <SidebarMenu key={child.uuid} menu={child} />);

      if (opened) {
        childrenContainer = (
          <ul className="nav nav-second-level">
            {childrenElements}
          </ul>
        );
      }
    }

    return (
      <li className={opened ? 'active' : ''}>
        <Link to={`/admin/projects/${projectId}/alerts`}>
          <i className="glyphicon glyphicon-bullhorn" />
          Alerts
          <ProjectAlertStats
            projectId={projectId}
            stats={projectAlertStats.evaluationSummary || {}}
          />
          {subMenus.length > 0
            && (
            <i
              className="glyphicon arrow"
              onClick={this.toggleChildren}
              role="link"
              tabIndex={-1}
            />
            )}
        </Link>
        {childrenContainer}
      </li>
    );
  }
}

AlertSidebarMenu.propTypes = {
  projectId: PropTypes.string.isRequired,
  subMenus: PropTypes.array.isRequired,
  opened: PropTypes.bool.isRequired,
  onOpenToggle: PropTypes.func.isRequired,
  projectAlertStats: PropTypes.object.isRequired,
  loadProjectAlertStats: PropTypes.func.isRequired,
  clearProjectAlertStats: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  projectAlertStats: state.projectAlertStats,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadProjectAlertStats,
  clearProjectAlertStats,
}, dispatch);

export default connect(mapStateToProps, mapDispatchToProps)(AlertSidebarMenu);
