import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Route, Switch } from 'react-router';
import { connect } from 'react-redux';

import SidebarMenu from './SidebarMenu';
import AlertSidebarMenu from './AlertSidebarMenu/AlertSidebarMenu';
import MutesSidebarMenu from './MutesSidebarMenu/MutesSidebarMenu';
import { MonitoringIcon } from '../../pages/old/components/ControlBar/MonitoringButton';

import Auth from '../../auth/Auth';
import { ADMIN } from '../../auth/Roles';
import SOLOMON_WEB_VERSION from '../../utils/version';
import { getMonitoringUiHost } from '../../pages/old/utils/graphToExpressionTransformations';

import './style.css';
import { isPreOrProdIntEnv } from '../../utils/env';

function constructMenu(title, link, icon = null, children = [], isLocal = true) {
  const uuid = Math.random();
  return {
    uuid,
    title: () => title,
    link: () => link,
    icon: () => icon,
    hasChild: () => !!children.length,
    children: () => children,
    isLocal: () => isLocal,
  };
}

function filterByName(array, pattern) {
  if (!pattern) {
    return array;
  }
  return array.filter((value) => (
    value.name && value.name.includes(pattern)
  ));
}

class ProjectMenu extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      filter: '',
      servicesOpened: false,
      clustersOpened: false,
      graphsOpened: false,
      dashboardsOpened: false,
      alertsOpened: false,
      channelsOpened: false,
      mutesOpened: false,
    };
  }

  onFilterChange = (event) => {
    this.setState({ filter: event.target.value });
  };

  onClustersToggle = () => {
    this.setState({ clustersOpened: !this.state.clustersOpened });
  };

  onServicesToggle = () => {
    this.setState({ servicesOpened: !this.state.servicesOpened });
  };

  onGraphsToggle = () => {
    this.setState({ graphsOpened: !this.state.graphsOpened });
  };

  onDashboardsToggle = () => {
    this.setState({ dashboardsOpened: !this.state.dashboardsOpened });
  };

  onAlertsToggle = () => {
    this.setState({ alertsOpened: !this.state.alertsOpened });
  };

  onChannelsToggle = () => {
    this.setState({ channelsOpened: !this.state.channelsOpened });
  };

  onMutesToggle = () => {
    this.setState({ mutesOpened: !this.state.mutesOpened });
  };

  render() {
    const {
      services, clusters, graphs, dashboards, alerts, channels, mutes,
    } = this.props;
    const { projectId } = this.props.match.params;
    const { filter, clustersOpened, servicesOpened } = this.state;

    const monitoringMenu = constructMenu(
      <strong>
        <MonitoringIcon />
        &nbsp;New: Monitoring UI
      </strong>,
      `https://${getMonitoringUiHost()}/projects/${projectId}?utm_source=solomon_menu_admin`,
      '',
      [],
      false,
    );

    const servicesMenu = constructMenu(
      'Services',
      `/admin/projects/${projectId}/services`,
      'cloud',
      filterByName(services, filter)
        .map((s) => constructMenu(
          s.name || s.id,
          `/admin/projects/${projectId}/services/${s.id}`,
        )),
    );

    const clustersMenu = constructMenu(
      'Clusters',
      `/admin/projects/${projectId}/clusters`,
      'equalizer',
      filterByName(clusters, filter)
        .map((c) => constructMenu(
          c.name || c.id,
          `/admin/projects/${projectId}/clusters/${c.id}`,
        )),
    );

    const shardsMenu = constructMenu(
      'Shards',
      `/admin/projects/${projectId}/shards`,
      'tasks',
    );

    const metricsMenu = constructMenu(
      'Metrics',
      `/admin/projects/${projectId}/metrics`,
      'random',
    );

    const strictValidationMenu = constructMenu(
      'Strict validation',
      `/admin/projects/${projectId}/strictValidation`,
      'warning-sign',
    );

    const graphsMenu = constructMenu(
      'Graphs',
      `/admin/projects/${projectId}/graphs`,
      'picture',
      filterByName(graphs, filter)
        .map((c) => constructMenu(
          c.name || c.id,
          `/admin/projects/${projectId}/graphs/${c.id}`,
        )),
    );

    const dashboardsMenu = constructMenu(
      'Dashboards',
      `/admin/projects/${projectId}/dashboards`,
      'dashboard',
      filterByName(dashboards, filter)
        .map((c) => constructMenu(
          c.name || c.id,
          `/admin/projects/${projectId}/dashboards/${c.id}`,
        )),
    );

    const alertSubMenus = filterByName(alerts, filter)
      .map((c) => constructMenu(
        c.name || c.id,
        `/admin/projects/${projectId}/alerts/${c.id}`,
      ));

    const channelsMenu = constructMenu(
      'Channels',
      `/admin/projects/${projectId}/channels`,
      'envelope',
      filterByName(channels, filter)
        .map((c) => constructMenu(
          c.name || c.id,
          `/admin/projects/${projectId}/channels/${c.id}`,
        )),
    );

    const mutesSubMenus = filterByName(mutes, filter)
      .map((m) => constructMenu(
        m.id,
        `/admin/projects/${projectId}/mutes/${m.id}`,
      ));

    const projectMenu = constructMenu(
      'Project menu',
      `/admin/projects/${projectId}/menu`,
      'list-alt',
    );

    return (
      <ul className="nav">
        <li className="sidebar-search">
          <input
            type="search"
            className="form-control"
            placeholder="Search..."
            onChange={this.onFilterChange}
          />
        </li>
        {isPreOrProdIntEnv() && (
          <SidebarMenu opened={false} menu={monitoringMenu} />
        )}
        <SidebarMenu
          opened={clustersOpened || !!filter}
          menu={clustersMenu}
          onOpenToggle={this.onClustersToggle}
        />
        <SidebarMenu
          opened={servicesOpened || !!filter}
          menu={servicesMenu}
          onOpenToggle={this.onServicesToggle}
        />
        <SidebarMenu menu={shardsMenu} />
        <AlertSidebarMenu
          projectId={projectId}
          subMenus={alertSubMenus}
          opened={this.state.alertsOpened || !!filter}
          onOpenToggle={this.onAlertsToggle}
        />
        <MutesSidebarMenu
          projectId={projectId}
          subMenus={mutesSubMenus}
          opened={this.state.mutesOpened || !!filter}
          onOpenToggle={this.onMutesToggle}
        />
        <SidebarMenu
          menu={channelsMenu}
          opened={this.state.channelsOpened || !!filter}
          onOpenToggle={this.onChannelsToggle}
        />
        <SidebarMenu
          opened={this.state.graphsOpened || !!filter}
          menu={graphsMenu}
          onOpenToggle={this.onGraphsToggle}
        />
        <SidebarMenu
          opened={this.state.dashboardsOpened || !!filter}
          menu={dashboardsMenu}
          onOpenToggle={this.onDashboardsToggle}
        />
        <SidebarMenu menu={metricsMenu} />
        <SidebarMenu menu={strictValidationMenu} />
        <SidebarMenu menu={projectMenu} />
      </ul>
    );
  }
}

ProjectMenu.propTypes = {
  services: PropTypes.array.isRequired,
  clusters: PropTypes.array.isRequired,
  graphs: PropTypes.array.isRequired,
  dashboards: PropTypes.array.isRequired,
  alerts: PropTypes.array.isRequired,
  channels: PropTypes.array.isRequired,
  mutes: PropTypes.array.isRequired,
  match: PropTypes.object.isRequired,
};

const ConnectedProjectMenu = connect(
  (state) => ({
    services: state.servicesPage.result || [],
    clusters: state.clustersPage.result || [],
    graphs: state.graphsPage.result || [],
    dashboards: state.dashboardsPage.result || [],
    alerts: state.alertsPage.items || [],
    channels: state.channelsPage.items || [],
    mutes: state.mutesPage.items || [],
  }),
)(ProjectMenu);

const RootMenu = () => (
  <ul className="nav">
    {isPreOrProdIntEnv() && (
      <SidebarMenu
        menu={constructMenu(
          <strong>
            <MonitoringIcon />
            &nbsp;New: Monitoring UI
          </strong>,
          `https://${getMonitoringUiHost()}/?utm_source=solomon_menu_admin`,
          '',
          [],
          false,
        )}
      />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Service Providers', '/admin/serviceProviders', 'cloud')} />
    )}
    <SidebarMenu menu={constructMenu('Projects', '/admin/projects', 'folder-open')} />
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Clusters', '/admin/clusters', 'equalizer')} />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Services', '/admin/services', 'cloud')} />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Shards', '/admin/shards', 'tasks')} />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Graphs', '/admin/graphs', 'picture')} />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Dashboards', '/admin/dashboards', 'dashboard')} />
    )}
    {Auth.hasRole(ADMIN) && (
      <SidebarMenu menu={constructMenu('Staff Only', '/staffOnly', 'user', [], false)} />
    )}
    <SidebarMenu menu={constructMenu('Preferences', '/admin/preferences', 'cog')} />
  </ul>
);

const Sidebar = () => (
  <div className="sidebar" id="sidebar">
    <div className="sidebar-menu nav-collapse">
      <Switch>
        <Route path="/admin/projects/new" component={RootMenu} />
        <Route path="/admin/projects/:projectId" component={ConnectedProjectMenu} />
        <Route component={RootMenu} />
      </Switch>
    </div>
    <div className="sidebar-footer">
      <small>
        {SOLOMON_WEB_VERSION}
      </small>
    </div>
  </div>
);

export default Sidebar;
