import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import './style.css';

const EI_NONE = -1;
const EI_NEW = -2;
const EI_CLONE = -3;

const CI_NONE = -1;

class TableEditor extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { editIndex: EI_NONE, cloneIndex: CI_NONE };
  }

  onDelete = (index) => {
    this.props.onDelete(index);
    this.clearState();
  };

  onUpdate = (index, rule) => {
    this.props.onUpdate(index, rule);
    this.clearState();
  };

  onMove = (index, direction) => {
    if (!this.props.onMove) {
      return;
    }
    this.props.onMove(index, direction);
    this.clearState();
  };

  onClone = (index) => {
    this.setState({ editIndex: EI_CLONE, cloneIndex: index });
  }

  onRowClick = (index) => {
    this.setState({ editIndex: index });
  };

  onAddClick = (event) => {
    event.preventDefault();
    this.setState({ editIndex: EI_NEW });
  };

  clearState = () => {
    this.setState({ editIndex: EI_NONE, cloneIndex: CI_NONE });
  };

  render() {
    const {
      columns, values, readonlyRow, editableRow, readOnly, additional = {},
    } = this.props;
    const { editIndex, cloneIndex } = this.state;

    /* eslint-disable react/no-array-index-key */
    const ths = columns.map((column, index) => <th key={index}>{column}</th>);

    const trs = values.map((value, index) => {
      if (editIndex === index) {
        return React.createElement(editableRow, {
          key: index,
          onOk: this.onUpdate,
          onCancel: this.clearState,
          index,
          value,
          readOnly,
          ...additional,
        });
      }
      return React.createElement(readonlyRow, {
        key: index,
        onClick: this.onRowClick,
        onDelete: this.onDelete,
        onMove: this.onMove,
        onClone: this.onClone,
        index,
        value,
        readOnly,
        ...additional,
      });
    });
    /* eslint-enable */

    let newTr = null;
    if (editIndex === EI_NEW) {
      newTr = React.createElement(editableRow, {
        key: EI_NEW,
        onOk: this.onUpdate,
        onCancel: this.clearState,
        index: EI_NEW,
        ...additional,
      });
    } else if (editIndex === EI_CLONE) {
      newTr = React.createElement(editableRow, {
        key: EI_NEW,
        onOk: this.onUpdate,
        onCancel: this.clearState,
        index: EI_NEW,
        value: values[cloneIndex],
        ...additional,
      });
    }

    const newItemButton = (
      <button type="button" className="btn btn-default" onClick={this.onAddClick}>
        <i className="glyphicon glyphicon-plus" />
      </button>
    );

    if (values.length === 0 && !newTr && !readOnly) {
      return newItemButton;
    }

    return (
      <div>
        <div className={this.props.responsive ? 'table-responsive' : ''}>
          <table className="table table-condensed table-hover table-row-pointer">
            <thead>
              <tr>
                {ths}
                <th />
              </tr>
            </thead>
            <tbody>
              {trs}
              {newTr}
            </tbody>
          </table>
        </div>
        {!readOnly && (
          <div className="action-row">
            {newItemButton}
          </div>
        )}
      </div>
    );
  }
}

TableEditor.propTypes = {
  columns: PropTypes.arrayOf(PropTypes.string).isRequired,
  values: PropTypes.array.isRequired,
  readonlyRow: PropTypes.func.isRequired,
  editableRow: PropTypes.func.isRequired,
  responsive: PropTypes.bool,
  onDelete: PropTypes.func.isRequired,
  onUpdate: PropTypes.func.isRequired,
  onMove: PropTypes.func,
  readOnly: PropTypes.bool,
  additional: PropTypes.object,
};

TableEditor.defaultProps = {
  onMove: null,
  responsive: true,
  readOnly: false,
  additional: {},
};

export default TableEditor;
