import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

const NO_LIMIT = -1;

class TableViewer extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { limit: props.limit };
  }

  onShowMoreClick = (event) => {
    event.preventDefault();
    this.setState({ limit: NO_LIMIT });
  };

  onShowLessClick = (event) => {
    event.preventDefault();
    this.setState({ limit: this.props.limit });
  };

  onAddClick = (event) => {
    event.preventDefault();
    this.props.onAdd();
  };

  render() {
    const {
      columns, values, row, onDelete, onAdd, onRowClick,
    } = this.props;
    const { limit } = this.state;
    const needLimit = (limit !== NO_LIMIT && limit < values.length);
    const wasLimited = (limit !== this.props.limit);

    // eslint-disable-next-line react/no-array-index-key
    const ths = columns.map((column, index) => <th key={index}>{column}</th>);

    const limitedValues = needLimit ? values.slice(0, limit) : values;
    const trs = limitedValues.map((value, index) => React.createElement(row, {
      // eslint-disable-next-line react/no-array-index-key
      key: index,
      index,
      value,
      onClick: onRowClick,
      onDelete,
    }));

    const showMoreTr = needLimit ? (
      <tr>
        <td colSpan={ths.length}>
          <a href="#" onClick={this.onShowMoreClick}>
            Show
            {' '}
            {values.length - limit}
            {' '}
            more...
          </a>
        </td>
      </tr>
    )
      : null;

    const showLessTr = wasLimited ? (
      <tr>
        <td colSpan={ths.length}>
          <a href="#" onClick={this.onShowLessClick}>Show less...</a>
        </td>
      </tr>
    )
      : null;

    const addButton = (
      <button type="button" className="btn btn-default" onClick={this.onAddClick}>
        <i className="glyphicon glyphicon-plus" />
      </button>
    );

    const addTr = onAdd ? (
      <tr className="action-row">
        <td colSpan={ths.length}>
          {addButton}
        </td>
      </tr>
    )
      : null;

    let tableClassName = 'table table-condensed';
    if (onRowClick) {
      tableClassName += ' table-hover table-row-pointer';
    }

    if (values.length === 0) {
      return onAdd ? addButton : (<span>No items found</span>);
    }

    return (
      <div className="table-responsive">
        <table className={tableClassName}>
          <thead>
            <tr>{ths}</tr>
          </thead>
          <tbody>
            {trs}
            {showMoreTr}
            {showLessTr}
            {addTr}
          </tbody>
        </table>
      </div>
    );
  }
}

TableViewer.propTypes = {
  columns: PropTypes.arrayOf(PropTypes.string).isRequired,
  values: PropTypes.array.isRequired,
  row: PropTypes.func.isRequired,
  limit: PropTypes.number,
  onDelete: PropTypes.func,
  onAdd: PropTypes.func,
  onRowClick: PropTypes.func,
};

TableViewer.defaultProps = {
  limit: NO_LIMIT,
  onDelete: null,
  onAdd: null,
  onRowClick: null,
};

export default TableViewer;
