import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

import './style.css';

export const Tab = ({ label, active, onClick }) => (
  <li className={active ? 'active' : ''}>
    {/* eslint-disable-next-line jsx-a11y/anchor-is-valid */}
    <a href="#" onClick={onClick}>{label}</a>
  </li>
);

Tab.propTypes = {
  label: PropTypes.string.isRequired,
  active: PropTypes.bool.isRequired,
  onClick: PropTypes.func.isRequired,
};

export const TabPane = ({ label, children }) => (
  <div label={label}>
    {children}
  </div>
);

TabPane.propTypes = {
  label: PropTypes.string.isRequired,
  children: PropTypes.node.isRequired,
};

export class Tabs extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      selected: props.memState === undefined || props.memState.defaultValue
        ? props.selected : props.memState.selected,
    };
  }

  handleClick = (index, event) => {
    event.preventDefault();
    this.setState({ selected: index });
    if (!this.props.memState.defaultValue) {
      this.props.memState.selected = index;
    }
  };

  _verticalTabs = (tabs, tabPane) => (
    <div className="row">
      <div className="col-xs-3">
        <ul className="nav nav-pills nav-stacked">{tabs}</ul>
      </div>
      <div className="col-xs-9">{tabPane}</div>
    </div>
  );

  _horizontalTabs = (tabs, tabPane) => (
    <div>
      <ul className="nav nav-tabs">{tabs}</ul>
      {tabPane}
    </div>
  );

  render() {
    const { vertical, children } = this.props;
    const { selected } = this.state;

    const filteredChidlren = children.filter(Boolean);

    const tabs = filteredChidlren.map((child, index) => (
      /* eslint-disable react/no-array-index-key */
      <Tab
        key={index}
        label={child.props.label}
        active={selected === index}
        onClick={(event) => this.handleClick(index, event)}
      />
      /* eslint-enable */
    ));

    const tabPane = filteredChidlren[selected];

    return (
      <div className="tabs">
        {vertical
          ? this._verticalTabs(tabs, tabPane)
          : this._horizontalTabs(tabs, tabPane)}
      </div>
    );
  }
}

Tabs.propTypes = {
  vertical: PropTypes.bool,
  selected: PropTypes.number,
  children: PropTypes.arrayOf(
    PropTypes.oneOfType([PropTypes.element, PropTypes.bool]),
  ).isRequired,
  memState: PropTypes.object,
};

Tabs.defaultProps = {
  vertical: false,
  selected: 0,
  memState: {
    selected: 0,
    defaultValue: true,
  },
};
