import React from 'react';
import PropTypes from 'prop-types';

import { Tab, Tabs } from '../Tabs';

class TabsWithUrl extends Tabs {
  constructor(props) {
    super(props);
    this.state = { activeTabId: props.activeTabId };
  }

  static getDerivedStateFromProps(props, state) {
    if (state.ownUpdate) {
      return { ownUpdate: false };
    }

    return { activeTabId: props.activeTabId };
  }

  handleClick = (tabId, event) => {
    event.preventDefault();
    event.target.blur();

    this.setState({ activeTabId: tabId, ownUpdate: true });
    this.props.setActiveTab(tabId);
  };

  _labelToId = (label) => label.trim().toLowerCase().replace(' ', '-');

  render() {
    const { vertical } = this.props;
    const { activeTabId } = this.state;

    const children = this.props.children.filter(Boolean);
    const tabs = [];
    let active = -1;
    const selectFirstTab = activeTabId.length === 0;

    children.forEach((child, index) => {
      /* eslint-disable react/no-array-index-key */
      const tabId = this._labelToId(child.props.label);
      if (!selectFirstTab && activeTabId === tabId) {
        active = index;
      }

      tabs.push(<Tab
        key={index}
        label={child.props.label}
        active={selectFirstTab ? (index === 0) : (index === active)}
        onClick={(event) => this.handleClick(tabId, event)}
      />);
      /* eslint-enable */
    });

    const tabPane = children[selectFirstTab ? 0 : active];

    return (
      <div className="tabs">
        {vertical
          ? this._verticalTabs(tabs, tabPane)
          : this._horizontalTabs(tabs, tabPane)}
      </div>
    );
  }
}

TabsWithUrl.propTypes = {
  vertical: PropTypes.bool,
  activeTabId: PropTypes.string,
  children: PropTypes.arrayOf(
    PropTypes.oneOfType([PropTypes.element, PropTypes.bool]),
  ).isRequired,
  setActiveTab: PropTypes.func.isRequired,
};

TabsWithUrl.defaultProps = {
  vertical: false,
  activeTabId: '',
};

export default TabsWithUrl;
