/* eslint-disable react/no-array-index-key */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { Link } from 'react-router-dom';

import './style.css';

export const TILE_SHAPE = PropTypes.shape({
  labels: PropTypes.array.isRequired,
  values: PropTypes.array.isRequired,
  url: PropTypes.string.isRequired,
  color: PropTypes.string.isRequired,
  mute: PropTypes.string.isRequired,
});

export const TILE_STYLE_SHAPE = PropTypes.shape({
  tileWidth: PropTypes.number.isRequired,
  tileHeight: PropTypes.number.isRequired,
  tileMargin: PropTypes.number.isRequired,
  tilePadding: PropTypes.number.isRequired,
  labelSize: PropTypes.number.isRequired,
  valueSize: PropTypes.number.isRequired,
});

export class Tile extends PureComponent {
  render() {
    const { style, isTargetBlank } = this.props;
    const {
      labels, values, url, color, mute,
    } = this.props.value;

    const hasStyle = style !== null;

    const hideLabels = hasStyle && style.labelSize === 0;
    const hideValues = hasStyle && style.valueSize === 0;

    let tileStyle = {};
    let innerTileStyle = {};
    let labelStyle = {};
    let valueStyle = {};

    if (hasStyle) {
      tileStyle = { margin: style.tileMargin };

      innerTileStyle = {
        width: style.tileWidth,
        minHeight: style.tileHeight,
        padding: style.tilePadding,
      };

      if (!style.labelSize && !style.valueSize) {
        innerTileStyle.padding = 0;
      }

      const innerWidth = style.tileWidth - (2 * style.tilePadding);
      labelStyle = { fontSize: style.labelSize, width: innerWidth };
      valueStyle = { fontSize: style.valueSize, width: innerWidth };
    }

    const tooltipTitle = [...labels, ...values].join('\n');

    return (
      <li className="tile" style={tileStyle}>
        <Link
          className={`tile__inner tile__inner_${color} tile__inner_${mute}`}
          to={url}
          target={isTargetBlank ? '_blank' : '_self'}
          rel="noopener noreferrer"
          title={tooltipTitle}
          style={innerTileStyle}
        >
          {!hideLabels && labels.map((label, index) => (
            <div className="tile__label" key={`label-${index}`} style={labelStyle}>
              {label || '-'}
            </div>
          ))}
          {!hideValues && values.map((value, index) => (
            <div className="tile__value" key={`value-${index}`} style={valueStyle}>
              {value || '-'}
            </div>
          ))}
        </Link>
      </li>
    );
  }
}

Tile.propTypes = {
  value: TILE_SHAPE.isRequired,
  style: TILE_STYLE_SHAPE,
  isTargetBlank: PropTypes.bool.isRequired,
};

Tile.defaultProps = {
  style: null,
};

export default {};
