/* eslint-disable no-plusplus,no-unused-vars,max-len,no-nested-ternary */

/* Area constants */
const AREA_PADDING = 10;

/* Tile constants */
const DEFAULT_TILE_WIDTH = 280;
const DEFAULT_TILE_HEIGHT = 120;

const NORMAL_LINE_HEIGHT = 1.2;

const MIN_LABEL_FONT_SIZE = 9;
const MAX_LABEL_FONT_SIZE = 18;

const MIN_VALUE_FONT_SIZE = 12;
const MAX_VALUE_FONT_SIZE = 36;

/* Tile modes */
const TM_WIDE_RECTANGLE = 0;
const TM_BIG_SQUARE = 2;
const TM_LITTLE_SQUARE = 3;

/* Tile mode constants */
const BIG_TILE_MARGIN = 2;
const LITTLE_TILE_MARGIN = 0;

const MIN_TILE_AS_RECTANGLE_SIZE = 30;
const MAX_TILE_AS_LITTLE_SQUARE_SIZE = 60;
const MIN_TILE_AS_LITTLE_SQUARE_SIZE = 5;
const MAX_TILE_AS_BIG_SQUARE_SIZE = 280;

/* Padding constants */
const TILE_PADDING = 15;
const TILE_PADDING_RATIO = TILE_PADDING / DEFAULT_TILE_WIDTH;

/* Singlestat constants */
const MAX_SINGLESTAT_SIZE = 2 * MAX_TILE_AS_BIG_SQUARE_SIZE;
const MAX_SINGLESTAT_LABEL_FONT_SIZE = 3 * MAX_LABEL_FONT_SIZE;
const MAX_SINGLESTAT_VALUE_FONT_SIZE = 3 * MAX_VALUE_FONT_SIZE;

let canvasContext = {};

if (process.env.NODE_ENV !== 'test') {
  const canvas = document.createElement('canvas');
  canvasContext = canvas.getContext('2d');
}

function computeMaxValue(values) {
  return values.length === 0 ? 0 : values.reduce((a, b) => Math.max(a, b));
}

function limitMaxValue(value, max) {
  return value > max ? max : value;
}

function limitMinMaxValue(value, min, max) {
  return value > max ? max : value < min ? min : value;
}

function checkThatTilesAreFitted(
  areaWidth, areaHeight,
  tileWidth, tileHeight, tileMargin,
  tilesCount,
) {
  const doubleTileMarginWithBorder = (tileMargin + 1) * 2;

  const widthForTiles = areaWidth - (2 * AREA_PADDING);

  const colCount = Math.floor(widthForTiles / (tileWidth + doubleTileMarginWithBorder));

  const rowCount = Math.ceil(tilesCount / colCount);

  const actualAreaHeight = (AREA_PADDING * 2) + (rowCount * (tileHeight + doubleTileMarginWithBorder));

  return actualAreaHeight > 0 && actualAreaHeight < areaHeight;
}

function computeTileSizesByComplexParams(
  areaWidth, areaHeight,
  startTileWidth, startTileHeight,
  minTileSize,
  tileMargin,
  tilesCount,
) {
  let tileWidth = startTileWidth;
  let tileHeight = startTileHeight;

  // eslint-disable-next-line max-len
  while (!checkThatTilesAreFitted(areaWidth, areaHeight, tileWidth, tileHeight, tileMargin, tilesCount)) {
    if (tileWidth <= minTileSize || tileHeight < minTileSize) {
      return null;
    }

    if (startTileHeight <= startTileWidth) {
      --tileHeight;
      tileWidth = Math.floor((tileHeight * startTileWidth) / startTileHeight);
    } else {
      --tileWidth;
      tileHeight = Math.floor((tileWidth * startTileHeight) / startTileWidth);
    }
  }

  return { tileWidth, tileHeight };
}

function computeWideRectangleTileSizes(areaWidth, areaHeight, tileMargin, tilesCount) {
  return computeTileSizesByComplexParams(
    areaWidth, areaHeight,
    DEFAULT_TILE_WIDTH, DEFAULT_TILE_HEIGHT,
    MIN_TILE_AS_RECTANGLE_SIZE,
    tileMargin,
    tilesCount,
  );
}

function computeBigSquareTileSizes(areaWidth, areaHeight, tileMargin, tilesCount) {
  return computeTileSizesByComplexParams(
    areaWidth, areaHeight,
    MAX_TILE_AS_BIG_SQUARE_SIZE, MAX_TILE_AS_BIG_SQUARE_SIZE,
    MAX_TILE_AS_LITTLE_SQUARE_SIZE,
    tileMargin,
    tilesCount,
  );
}

function computeLittleSquareTileSizes(areaWidth, areaHeight, tileMargin, tilesCount) {
  return computeTileSizesByComplexParams(
    areaWidth, areaHeight,
    MAX_TILE_AS_LITTLE_SQUARE_SIZE, MAX_TILE_AS_LITTLE_SQUARE_SIZE,
    MIN_TILE_AS_LITTLE_SQUARE_SIZE,
    tileMargin,
    tilesCount,
  );
}

function computeTileSizesForArea(areaWidth, areaHeight, tilesCount, isSquareTitles) {
  let tileSizes = null;
  let tileMargin = BIG_TILE_MARGIN;
  let mode;

  if (isSquareTitles) {
    mode = TM_BIG_SQUARE;
    tileSizes = computeBigSquareTileSizes(areaWidth, areaHeight, tileMargin, tilesCount);
  } else {
    mode = TM_WIDE_RECTANGLE;
    tileSizes = computeWideRectangleTileSizes(areaWidth, areaHeight, tileMargin, tilesCount);
  }

  if (tileSizes === null) {
    mode = TM_LITTLE_SQUARE;
    tileMargin = LITTLE_TILE_MARGIN;
    tileSizes = computeLittleSquareTileSizes(areaWidth, areaHeight, tileMargin, tilesCount);

    if (tileSizes === null) {
      tileSizes = {
        tileWidth: MIN_TILE_AS_LITTLE_SQUARE_SIZE,
        tileHeight: MIN_TILE_AS_LITTLE_SQUARE_SIZE,
      };
    }
  }

  return { ...tileSizes, mode, tileMargin };
}

function fitTitleByWidth(width, titleSize, minSize, titleLen) {
  if (titleSize < minSize) {
    return 0;
  }

  let str = '';
  for (let i = 0; i < titleLen; ++i) {
    str += 'X';
  }

  // eslint-disable-next-line no-param-reassign
  canvasContext.font = `${titleSize}px serif`;

  const measure = canvasContext.measureText(str);

  const valueWidth = measure.width;

  let resultSize = titleSize;

  if (valueWidth > width) {
    resultSize = (titleSize * width) / valueWidth;
    resultSize = resultSize >= minSize ? resultSize : 0;
  }

  return resultSize;
}

function fitInnerTitleSizes(tileWidth, tileHeight, tilePadding, labelsCount, valuesCount, labelLen, valueLen) {
  let labelSize;
  let valueSize;

  const innerHeight = (tileHeight - (2 * tilePadding)) / NORMAL_LINE_HEIGHT;
  const innerWidth = tileWidth - (2 * tilePadding);

  if (labelsCount > 0 && valuesCount > 0) {
    const alpha = innerHeight / ((MAX_LABEL_FONT_SIZE * labelsCount) + (MAX_VALUE_FONT_SIZE * valuesCount));
    labelSize = Math.ceil(alpha * MAX_LABEL_FONT_SIZE);
    valueSize = Math.ceil(alpha * MAX_VALUE_FONT_SIZE);
    valueSize = fitTitleByWidth(innerWidth, valueSize, MIN_VALUE_FONT_SIZE, valueLen);

    if (labelSize === 0 || valueSize === 0) {
      labelSize = 0;
      valueSize = Math.ceil(innerHeight / valuesCount);
    }
  } else if (labelsCount === 0 && valuesCount > 0) {
    labelSize = 0;
    valueSize = Math.ceil(innerHeight / valuesCount);
  } else if (labelsCount > 0 && valuesCount === 0) {
    labelSize = Math.ceil(innerHeight / labelsCount);
    valueSize = 0;
  } else {
    labelSize = 0;
    valueSize = 0;
  }

  return { labelSize, valueSize };
}

function computeMaxTitleLenInTile(tile, name) {
  const titles = tile[name];
  if (titles.length === 0) {
    return 0;
  }

  if (titles.length === 1) {
    return titles[0].length;
  }

  return titles.map((title) => title.length).reduce((a, b) => Math.max(a, b));
}

function computeMaxTitleLen(tiles, name) {
  if (tiles.length === 0) {
    return 0;
  }

  return tiles
    .map((tile) => computeMaxTitleLenInTile(tile, name))
    .reduce((a, b) => Math.max(a, b));
}

export function computeFitTileStyle(areaWidth, areaHeight, tiles) {
  const tilesCount = tiles.length;

  const labelsCount = computeMaxValue(tiles.map((tile) => tile.labels.length));
  const valuesCount = computeMaxValue(tiles.map((tile) => tile.values.length));

  const isSquareTitles = labelsCount + valuesCount >= 5;

  const {
    mode, tileWidth, tileHeight, tileMargin,
  } = computeTileSizesForArea(areaWidth, areaHeight, tilesCount, isSquareTitles);

  const tilePadding = Math.min(TILE_PADDING, Math.ceil(tileWidth * TILE_PADDING_RATIO));

  let labelSize;
  let valueSize;

  if (mode === TM_LITTLE_SQUARE) {
    labelSize = 0;
    valueSize = 0;
  } else {
    const maxLabelLen = Math.ceil(0.75 * computeMaxTitleLen(tiles, 'labels'));
    const maxValueLen = Math.ceil(computeMaxTitleLen(tiles, 'values'));

    const size = fitInnerTitleSizes(tileWidth, tileHeight, tilePadding, labelsCount, valuesCount, maxLabelLen, maxValueLen);

    labelSize = limitMaxValue(size.labelSize, MAX_LABEL_FONT_SIZE);
    valueSize = limitMaxValue(size.valueSize, MAX_VALUE_FONT_SIZE);
  }

  return {
    tileWidth, tileHeight, tileMargin, tilePadding, labelSize, valueSize,
  };
}

export function computeFitSinglestatStyle(areaWidth, areaHeight, tile) {
  const labelsCount = tile.labels.length;
  const valuesCount = tile.values.length;

  let tileWidth = areaWidth - (2 * (AREA_PADDING + 1));
  tileWidth = limitMinMaxValue(tileWidth, MIN_TILE_AS_LITTLE_SQUARE_SIZE, MAX_SINGLESTAT_SIZE);
  let tileHeight = areaHeight - (2 * (AREA_PADDING + 1));
  tileHeight = limitMinMaxValue(tileHeight, MIN_TILE_AS_LITTLE_SQUARE_SIZE, MAX_SINGLESTAT_SIZE);

  const minTileSize = Math.min(tileWidth, tileHeight);

  let isLittleSquare;
  let tileMargin;

  if (minTileSize >= MAX_TILE_AS_LITTLE_SQUARE_SIZE) {
    isLittleSquare = false;
    tileMargin = BIG_TILE_MARGIN;
  } else {
    isLittleSquare = true;
    tileMargin = LITTLE_TILE_MARGIN;
  }

  const tilePadding = Math.min(TILE_PADDING, Math.ceil(minTileSize * TILE_PADDING_RATIO));

  let labelSize;
  let valueSize;

  if (isLittleSquare) {
    labelSize = 0;
    valueSize = 0;
  } else {
    const maxLabelLen = computeMaxTitleLenInTile(tile, 'labels');
    const maxValueLen = computeMaxTitleLenInTile(tile, 'values');

    const size = fitInnerTitleSizes(tileWidth, tileHeight, tilePadding, labelsCount, valuesCount, maxLabelLen, maxValueLen);

    labelSize = limitMaxValue(size.labelSize, MAX_SINGLESTAT_LABEL_FONT_SIZE);
    valueSize = limitMaxValue(size.valueSize, MAX_SINGLESTAT_VALUE_FONT_SIZE);
  }

  return {
    tileWidth, tileHeight, tileMargin, tilePadding, labelSize, valueSize,
  };
}

export default {};
