import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { formatDurationSecs, parseDuration } from '../../utils/duration';

const CustomDurationButton = ({ duration, onClick }) => (
  <span
    className="label label-default"
    role="button"
    tabIndex={-1}
    style={{ marginRight: '3px' }}
    onClick={(event) => { event.preventDefault(); onClick(duration); }}
  >
    {duration}
  </span>
);

CustomDurationButton.propTypes = {
  duration: PropTypes.string.isRequired,
  onClick: PropTypes.func.isRequired,
};

class DurationInput extends PureComponent {
  constructor(props) {
    super(props);
    this.state = { value: formatDurationSecs(props.value), isValid: true };
  }

  componentDidUpdate(prevProps) {
    if (this.props.value !== prevProps.value) {
      this.setState({ value: formatDurationSecs(this.props.value), isValid: true });
    }
  }

  handleInputChange = (event) => {
    event.preventDefault();

    const { value } = event.target;

    this.handleValueChange(value);
  };

  handleValueChange = (value) => {
    const millis = parseDuration(value, true);

    const isValid = millis !== null;

    this.setState({ value, isValid });

    if (isValid) {
      this.props.onChange(millis);
    }
  };

  render() {
    const { name, customValues } = this.props;

    return (
      <div className={this.state.isValid ? '' : ' has-error'}>
        <input
          className="form-control"
          name={name}
          value={this.state.value}
          onChange={this.handleInputChange}
        />
        <div className="help-block">
          {customValues.map((value) => (
            <CustomDurationButton key={value} duration={value} onClick={this.handleValueChange} />
          ))}
        </div>
      </div>
    );
  }
}

DurationInput.propTypes = {
  name: PropTypes.string.isRequired,
  value: PropTypes.number,
  onChange: PropTypes.func.isRequired,
  customValues: PropTypes.arrayOf(PropTypes.string),
};

DurationInput.defaultProps = {
  value: null,
  customValues: ['5m', '10m', '30m', '1h'],
};

export default DurationInput;
