import React, { PureComponent, createRef } from 'react';
import PropTypes from 'prop-types';
import { DateTime } from 'react-datetime-bootstrap';
import flow from 'lodash/fp/flow';
import {
  addMilliseconds,
  formatISO,
  setHours,
  setMinutes,
  setSeconds,
  addDays,
} from 'date-fns/fp';
import nextMonday from 'date-fns/nextMonday';
import moment from 'moment';
import HelpButton from './HelpButton';
import { parseDuration } from '../../utils/duration';

import 'eonasdan-bootstrap-datetimepicker/build/css/bootstrap-datetimepicker.min.css';
import './FormDate.css';

const SHORTUC_MAP = {
  'Until tomorrow': (ref) => {
    const date = new Date(ref);
    return flow(addDays(1), setHours(10), setMinutes(0), setSeconds(0), formatISO)(date);
  },
  'Until Monday': (ref) => {
    const date = new Date(ref);
    return flow(nextMonday, setHours(10), setMinutes(0), setSeconds(0), formatISO)(date);
  },
};

const ShortcutButton = ({ duration, onClick }) => (
  <span
    className="label label-default"
    role="button"
    tabIndex={-1}
    style={{ marginRight: '3px' }}
    onClick={(event) => { event.preventDefault(); onClick(duration); }}
  >
    {duration}
  </span>
);

ShortcutButton.propTypes = {
  duration: PropTypes.string.isRequired,
  onClick: PropTypes.func.isRequired,
};

class FormDate extends PureComponent {
  static lastId = 0;

  static nextUniqId(prefix) {
    FormDate.lastId += 1;
    return prefix + FormDate.lastId;
  }

  constructor(props) {
    super(props);
    this.id = FormDate.nextUniqId(props.name);
    this.inputRef = createRef();
  }

  onChange = (value) => {
    this.props.onChange({
      target: this.inputRef.current.textInputElement,
      value,
    });
  }

  handleValueChange = (value) => {
    if (SHORTUC_MAP[value]) {
      this.onChange(SHORTUC_MAP[value](this.props.dateReference));
      return;
    }

    const parsedValue = parseDuration(value);
    const date = new Date(this.props.dateReference);
    this.onChange(formatISO(addMilliseconds(date, parsedValue)));
  };

  focus() {
    this.inputRef.current.focus();
  }

  render() {
    const {
      type, name, label, labelWidth, value, dateReference,
      extHelp, help, warning, placeholder, prefix, suffix, shortcuts, disabled,
    } = this.props;

    const format = 'YYYY-MM-DD HH:mm:ss';

    const valueParsed = value && moment(value).format(format);

    const inputEl = (
      <DateTime
        id={this.id}
        name={name}
        type={type}
        pickerOptions={{
          format,
        }}
        placeholder={placeholder}
        onChange={this.onChange}
        value={valueParsed}
        disabled={disabled}
        ref={this.inputRef}
      />
    );

    let inputGroup = inputEl;
    if (prefix || suffix) {
      inputGroup = (
        <div className="input-group">
          {prefix && <div className="input-group-addon">{prefix}</div>}
          {inputEl}
          {suffix && <div className="input-group-addon">{suffix}</div>}
        </div>
      );
    }

    return (
      <div className="form-group">
        <label className={`col-sm-${labelWidth} control-label`} htmlFor={this.id}>
          {label}
          <HelpButton label={label} help={extHelp} />
        </label>
        <div className={`col-sm-${12 - labelWidth}`}>
          {inputGroup}

          {shortcuts.length > 0 && dateReference && (
            <div className="help-block">
              {shortcuts.map((shortcut) => (
                <ShortcutButton
                  key={shortcut}
                  duration={shortcut}
                  onClick={this.handleValueChange}
                />
              ))}
            </div>
          )}

          {help && (
          <span className="help-block">
            {help}
            .
          </span>
          )}
          {warning && <div className="alert alert-warning">{warning}</div>}
        </div>
      </div>
    );
  }
}

FormDate.propTypes = {
  type: PropTypes.string.isRequired,
  dateReference: PropTypes.string,
  name: PropTypes.string.isRequired,
  label: PropTypes.string.isRequired,
  labelWidth: PropTypes.number,
  onChange: PropTypes.func.isRequired,
  value: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
  warning: PropTypes.oneOfType([PropTypes.string, PropTypes.node]),
  extHelp: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  help: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  placeholder: PropTypes.string,
  disabled: PropTypes.bool,
  prefix: PropTypes.string,
  suffix: PropTypes.string,
  shortcuts: PropTypes.arrayOf(PropTypes.string),
};

FormDate.defaultProps = {
  value: '',
  labelWidth: 2,
  extHelp: '',
  help: '',
  warning: '',
  suffix: '',
  prefix: '',
  placeholder: '',
  shortcuts: [],
  disabled: false,
  dateReference: undefined,
};

export default FormDate;
