import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import {
  formatDuration, formatDurationSecs, parseDuration, parseDurationSecs,
} from '../../utils/duration';

const CustomDurationButton = ({ duration, onClick }) => (
  <span
    className="label label-default"
    role="button"
    tabIndex={-1}
    style={{ marginRight: '3px' }}
    onClick={(event) => { event.preventDefault(); onClick(duration); }}
  >
    {duration}
  </span>
);

CustomDurationButton.propTypes = {
  duration: PropTypes.string.isRequired,
  onClick: PropTypes.func.isRequired,
};

class FormDurationInput extends PureComponent {
  constructor(props) {
    super(props);
    const formattedValue = props.format === 'seconds'
      ? formatDurationSecs(props.value)
      : formatDuration(props.value);
    this.state = { value: formattedValue, isValid: true };
  }

  componentDidUpdate(prevProps) {
    if (this.props.value !== prevProps.value
      || this.props.name !== prevProps.name
      || this.props.format !== prevProps.format) {
      const formattedValue = this.props.format === 'seconds'
        ? formatDurationSecs(this.props.value)
        : formatDuration(this.props.value);

      this.setState({ value: formattedValue, isValid: true });
    }
  }

  handleInputChange = (event) => {
    event.preventDefault();

    const { value } = event.target;

    this.handleValueChange(value);
  };

  handleValueChange = (value) => {
    const parsedValue = this.props.format === 'seconds'
      ? parseDurationSecs(value)
      : parseDuration(value);

    const isValid = parsedValue !== null;

    this.setState({ value, isValid });

    if (isValid) {
      this.props.onChange(parsedValue);
    }
  };

  render() {
    const {
      label, name, customValues, help, disabled,
    } = this.props;

    return (
      <div className={`form-group${this.state.isValid ? '' : ' has-error'}`}>
        {/* eslint-disable jsx-a11y/label-has-for */}
        <label className="col-sm-2 control-label">
          {label}
        </label>
        <div className="col-sm-10">
          <input
            className="form-control"
            name={name}
            value={this.state.value}
            onChange={this.handleInputChange}
            disabled={disabled}
          />
          {!disabled && (
          <div className="help-block">
            {customValues.map((value) => (
              <CustomDurationButton
                key={value}
                duration={value}
                onClick={this.handleValueChange}
              />
            ))}
          </div>
          )}
          {help && (
          <span className="help-block">
            {help}
            .
          </span>
          )}
        </div>
      </div>
    );
  }
}

FormDurationInput.propTypes = {
  label: PropTypes.string.isRequired,
  name: PropTypes.string.isRequired,
  format: PropTypes.string,
  value: PropTypes.number,
  onChange: PropTypes.func.isRequired,
  customValues: PropTypes.arrayOf(PropTypes.string),
  help: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  disabled: PropTypes.bool,
};

FormDurationInput.defaultProps = {
  format: '',
  value: null,
  customValues: ['5m', '10m', '30m', '1h'],
  help: '',
  disabled: false,
};

export default FormDurationInput;
