import React, { PureComponent, createRef } from 'react';
import PropTypes from 'prop-types';
import HelpButton from './HelpButton';

class FormInput extends PureComponent {
  static lastId = 0;

  static nextUniqId(prefix) {
    FormInput.lastId += 1;
    return prefix + FormInput.lastId;
  }

  constructor(props) {
    super(props);
    this.id = FormInput.nextUniqId(props.name);
    this.inputRef = createRef();
  }

  // hack to prevent input number value scrolling
  onWheel = (event) => {
    const { target } = event;
    if (target.type === 'number') {
      target.blur();
    }
  };

  focus() {
    this.inputRef.current.focus();
  }

  render() {
    const {
      type, name, label, labelWidth, value, onChange,
      extHelp, help, warning, placeholder, prefix, suffix, disabled,
    } = this.props;

    const inputEl = (
      <input
        id={this.id}
        name={name}
        className="form-control"
        type={type}
        placeholder={placeholder}
        onChange={onChange}
        onWheel={this.onWheel}
        value={value}
        disabled={disabled}
        ref={this.inputRef}
      />
    );

    let inputGroup = inputEl;
    if (prefix || suffix) {
      inputGroup = (
        <div className="input-group">
          {prefix && <div className="input-group-addon">{prefix}</div>}
          {inputEl}
          {suffix && <div className="input-group-addon">{suffix}</div>}
        </div>
      );
    }

    return (
      <div className="form-group">
        <label className={`col-sm-${labelWidth} control-label`} htmlFor={this.id}>
          {label}
          <HelpButton label={label} help={extHelp} />
        </label>
        <div className={`col-sm-${12 - labelWidth}`}>
          {inputGroup}
          {help && (
          <span className="help-block">
            {help}
            .
          </span>
          )}
          {warning && <div className="alert alert-warning">{warning}</div>}
        </div>
      </div>
    );
  }
}

FormInput.propTypes = {
  type: PropTypes.string.isRequired,
  name: PropTypes.string.isRequired,
  label: PropTypes.string.isRequired,
  labelWidth: PropTypes.number,
  onChange: PropTypes.func.isRequired,
  value: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
  warning: PropTypes.oneOfType([PropTypes.string, PropTypes.node]),
  extHelp: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  help: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  placeholder: PropTypes.string,
  prefix: PropTypes.string,
  suffix: PropTypes.string,
  disabled: PropTypes.bool,
};

FormInput.defaultProps = {
  value: '',
  labelWidth: 2,
  extHelp: '',
  help: '',
  warning: '',
  placeholder: '',
  prefix: '',
  suffix: '',
  disabled: false,
};

export default FormInput;
