/* eslint-disable jsx-a11y/no-autofocus,max-len */
import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { isCloud } from '../../utils/env';

const METRIC_NAME_TYPES = {
  sensor: 'sensor',
  path: 'path',
  metric: 'metric',
  name: 'name',
  signal: 'signal',
  custom: 'custom',
  none: 'none',
};

function findMetricNameType(value) {
  if (!value) {
    return METRIC_NAME_TYPES.none;
  }

  if (value === METRIC_NAME_TYPES.sensor) {
    return METRIC_NAME_TYPES.sensor;
  }

  if (value === METRIC_NAME_TYPES.path) {
    return METRIC_NAME_TYPES.path;
  }

  if (value === METRIC_NAME_TYPES.metric) {
    return METRIC_NAME_TYPES.metric;
  }

  if (value === METRIC_NAME_TYPES.signal) {
    return METRIC_NAME_TYPES.signal;
  }

  if (value === METRIC_NAME_TYPES.name) {
    return METRIC_NAME_TYPES.name;
  }

  return METRIC_NAME_TYPES.custom;
}

class FormMetricNameLabel extends PureComponent {
  constructor(props) {
    super(props);

    this.state = { type: findMetricNameType(props.value) };
  }

  onInputChange = (event) => {
    event.preventDefault();

    const { value } = event.target;

    this.props.onChange(value);
  };

  onTypeChange = (event) => {
    event.preventDefault();
    const type = event.target.name;
    this.setState({ type });
    if (type === METRIC_NAME_TYPES.none || type === METRIC_NAME_TYPES.custom) {
      this.props.onChange('');
    } else {
      this.props.onChange(type);
    }
  };

  render() {
    const {
      label, name, value, disabled, type, onlyMetricNameShards,
    } = this.props;

    if (disabled) {
      if (!value) {
        return null;
      }

      return (
        <div className="form-group">
          {/* eslint-disable jsx-a11y/label-has-for */}
          <label className="col-sm-2 control-label">
            {label}
          </label>
          <div className="col-sm-10">
            <input
              className="form-control"
              value={value}
              disabled
            />
            <div className="help-block">
              Send an email to
              {' '}
              <a href="mailto:solomon.yandex-team.ru">solomon@</a>
              {' '}
              mailing list or
              a message to
              {' '}
              <a href="https://nda.ya.ru/t/4XRL2d_13YLtZs">telegram chat</a>
              &nbsp; if you want to change metric name label in
              {' '}
              {type}
            </div>
          </div>
        </div>
      );
    }

    let help;

    const commonHelp = (
      <span>
        Required metric name label for
        {' '}
        {type}
        {' '}
        metrics.
        <br />
        It&#39;s neccessary to convert
        {' '}
        <code>sensor=&lt;metric_name&gt;, label1=&quot;value1&quot;, label2=&quot;value2&quot;</code>
        {' '}
        to
        {' '}
        <code>&lt;metric_name&gt;&#123;label1=&quot;value1&quot;, label2=&quot;value2&quot;&#125;</code>
        {' '}
        format.
        <br />
        This label must be required in pull/push data and existing metrics.
        <br />
        See
        {' '}
        <a href="https://st.yandex-team.ru/SOLOMON-3923" rel="noopener noreferrer" target="_blank">SOLOMON-3923</a>
        {' '}
        for more information.
      </span>
    );

    if (type === 'shard') {
      help = (
        <div className="help-block">
          {commonHelp}
          <p />
          <span className="text-danger">
            <i className="glyphicon glyphicon-warning-sign" />
            {' '}
            Don&apos; set this option except special case
            when shards in service have different metrics name labels!
          </span>
        </div>
      );
    } else {
      help = (
        <div className="help-block">
          {commonHelp}
        </div>
      );
    }

    let defaultLabelTitle;

    if (onlyMetricNameShards) {
      const defaultLabel = isCloud() ? 'name' : 'sensor';
      defaultLabelTitle = `Default (${defaultLabel})`;
    } else {
      defaultLabelTitle = 'None';
    }

    return (
      <div className="form-group">
        {/* eslint-disable jsx-a11y/label-has-for */}
        <label className="col-sm-2 control-label">
          {label}
        </label>
        <div className="col-sm-10">
          <div className="btn-toolbar">
            <div className="btn-group">
              {['sensor', 'path', 'metric', 'signal', 'name'].map((c) => (
                <button
                  key={c}
                  name={c}
                  type="button"
                  className={this.state.type === c ? 'btn btn-default active' : 'btn btn-default'}
                  onClick={this.onTypeChange}
                >
                  {c}
                </button>
              ))}
            </div>
            <div className="btn-group">
              <button
                type="button"
                name="custom"
                className={this.state.type === METRIC_NAME_TYPES.custom ? 'btn btn-default active' : 'btn btn-default'}
                onClick={this.onTypeChange}
              >
                Custom
              </button>
            </div>
            <div className="btn-group">
              <button
                type="button"
                name="none"
                className={this.state.type === METRIC_NAME_TYPES.none ? 'btn btn-default active' : 'btn btn-default'}
                onClick={this.onTypeChange}
              >
                {defaultLabelTitle}
              </button>
            </div>
          </div>
          {this.state.type === METRIC_NAME_TYPES.custom && (
            <input
              style={{ marginTop: '5px' }}
              className="form-control"
              name={name}
              value={this.state.value}
              onChange={this.onInputChange}
              autoFocus
            />
          )}
          {help}
        </div>
      </div>
    );
  }
}

FormMetricNameLabel.propTypes = {
  label: PropTypes.string.isRequired,
  name: PropTypes.string.isRequired,
  value: PropTypes.string,
  onChange: PropTypes.func.isRequired,
  disabled: PropTypes.bool,
  type: PropTypes.oneOf('project', 'service', 'shard').isRequired,
  onlyMetricNameShards: PropTypes.bool.isRequired,
};

FormMetricNameLabel.defaultProps = {
  value: '',
  disabled: false,
};

export default FormMetricNameLabel;
