import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';

class FormNumber extends PureComponent {
  static lastId = 0;

  static nextUniqId(prefix) {
    FormNumber.lastId += 1;
    return prefix + FormNumber.lastId;
  }

  constructor(props) {
    super(props);
    this.id = FormNumber.nextUniqId(props.name);
    this.state = { value: props.value };
  }

  componentDidUpdate(prevProps) {
    if (this.props.value !== prevProps.value) {
      this.setState({ value: this.props.value });
    }
  }

  handleChange = (event) => {
    event.preventDefault();

    const { value } = event.target;

    let parsedValue;
    let isValidValue;

    if (value) {
      parsedValue = this.props.type === 'int' ? parseInt(value, 10) : parseFloat(value);
      isValidValue = !isNaN(parsedValue);
    } else {
      parsedValue = value;
      isValidValue = true;
    }

    this.setState({ value });

    if (isValidValue) {
      this.props.onChange(event, parsedValue);
    }
  };

  // Need to disable unexpected value scrolling
  handleWheel = (event) => {
    event.preventDefault();
  };

  // Need to disable up and down arrows on keyboard
  handleKeyDown = (event) => {
    const key = event.keyCode;
    if (key === 38 || key === 40) {
      event.preventDefault();
    }
  };

  render() {
    const {
      name, label,
      help, placeholder, prefix, suffix, disabled,
    } = this.props;

    const inputEl = (
      <input
        id={this.id}
        name={name}
        className="form-control"
        type="text"
        placeholder={placeholder}
        value={this.state.value}
        disabled={disabled}
        onChange={this.handleChange}
        onWheel={this.handleWheel}
        onKeyDown={this.handleKeyDown}
      />
    );

    let inputGroup = inputEl;
    if (prefix || suffix) {
      inputGroup = (
        <div className="input-group">
          {prefix && <div className="input-group-addon">{prefix}</div>}
          {inputEl}
          {suffix && <div className="input-group-addon">{suffix}</div>}
        </div>
      );
    }

    return (
      <div className="form-group">
        <label className="col-sm-2 control-label" htmlFor={this.id}>
          {label}
        </label>
        <div className="col-sm-10">
          {inputGroup}
          {help && (
          <span className="help-block">
            {help}
            .
          </span>
          )}
        </div>
      </div>
    );
  }
}

FormNumber.propTypes = {
  name: PropTypes.string.isRequired,
  type: PropTypes.oneOf(['int', 'float']),
  label: PropTypes.string.isRequired,
  onChange: PropTypes.func.isRequired,
  value: PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
  help: PropTypes.oneOfType([PropTypes.string, PropTypes.element]),
  placeholder: PropTypes.string,
  prefix: PropTypes.string,
  suffix: PropTypes.string,
  disabled: PropTypes.bool,
};

FormNumber.defaultProps = {
  type: 'int',
  value: '',
  help: '',
  placeholder: '',
  prefix: '',
  suffix: '',
  disabled: false,
};

export default FormNumber;
