import React, { PureComponent } from 'react';
import { connect } from 'react-redux';
import FormGroup from '../../components/forms/FormGroup';

const DEFAULT_WARNING_GT_THRESHOLD = 0;
const DEFAULT_DANGER_GT_THRESHOLD = 25;

const WARNING_THRESHOLD_KEY = 'warningHealthThreshold';
const DANGER_THRESHOLD_KEY = 'dangerHealthThreshold';
const ALERT_HEATH_MUTED_KEY = 'alertHeathMuted';

function getThresholdOrDefault(key, defaultValue) {
  const propertyStr = window.localStorage.getItem(key);
  const threshold = parseInt(propertyStr, 10);
  return isNaN(threshold) ? defaultValue : threshold;
}

function tryToSetThresholds(warningValue, dangerValue) {
  if (isNaN(warningValue) || isNaN(dangerValue)) {
    return false;
  }

  if (warningValue < 0 || warningValue > 100
    || dangerValue < 0 || dangerValue > 100
    || warningValue > dangerValue) {
    return false;
  }

  window.localStorage.setItem(WARNING_THRESHOLD_KEY, warningValue);
  window.localStorage.setItem(DANGER_THRESHOLD_KEY, dangerValue);

  return true;
}

class AlertingHealthForm extends PureComponent {
  constructor(props) {
    super(props);
    this.state = {
      warningThreshold: getThresholdOrDefault(WARNING_THRESHOLD_KEY, DEFAULT_WARNING_GT_THRESHOLD),
      dangerThreshold: getThresholdOrDefault(DANGER_THRESHOLD_KEY, DEFAULT_DANGER_GT_THRESHOLD),
      muted: window.localStorage.getItem(ALERT_HEATH_MUTED_KEY) === 'true',
      isWarningThresholdValid: true,
      isDangerThresholdValid: true,
    };
  }

  handleWarningThresholdChange = (event) => {
    event.preventDefault();
    const warningThreshold = event.target.value;
    const result = tryToSetThresholds(warningThreshold, this.state.dangerThreshold);
    this.setState({ warningThreshold, isWarningThresholdValid: result });
  };

  handleDangerThresholdChange = (event) => {
    event.preventDefault();
    const dangerThreshold = event.target.value;
    const result = tryToSetThresholds(this.state.warningThreshold, dangerThreshold);
    this.setState({ dangerThreshold, isDangerThresholdValid: result });
  };

  handleMuteChange = (event) => {
    const muted = event.target.checked;
    window.localStorage.setItem(ALERT_HEATH_MUTED_KEY, `${muted}`);
    this.setState({ muted });
  };

  render() {
    return (
      <div className="form-horizontal">
        <FormGroup title="Graph health alerting settings">
          <div className={`form-group${this.state.isWarningThresholdValid ? '' : ' has-error'}`}>
            <label
              htmlFor="warning-threshold-input"
              className="col-sm-2 control-label"
            >
              Warning threshold:
            </label>
            <div className="col-sm-10">
              <input
                type="number"
                id="warning-threshold-input"
                name="warningThreshold"
                className="form-control"
                min="0"
                max="100"
                value={this.state.warningThreshold}
                onChange={this.handleWarningThresholdChange}
                disabled={this.state.muted}
              />
              <div className="help-block">
                Min target rate in percents to show warning alert
                <br />
                (f.e 10 - show warning alert for greater than 10% targets)
              </div>
            </div>
          </div>
          <div className={`form-group${this.state.isDangerThresholdValid ? '' : ' has-error'}`}>
            <label
              htmlFor="danger-threshold-input"
              className="col-sm-2 control-label"
            >
              Danger threshold:
            </label>
            <div className="col-sm-10">
              <input
                type="number"
                id="danger-threshold-input"
                name="dangerThreshold"
                className="form-control"
                min="0" max="100"
                value={this.state.dangerThreshold}
                onChange={this.handleDangerThresholdChange}
                disabled={this.state.muted}
              />
              <div className="help-block">
                Min target rate in percents to show danger alert
                <br />
                (f.e. 50 - show danger alert for greater than 50% targets)
              </div>
            </div>
          </div>
          <div className="form-group">
            <div className="col-sm-offset-2 col-sm-10">
              <label htmlFor="mute-health-alert-checkbox">
                <input
                  type="checkbox"
                  id="mute-health-alert-checkbox"
                  checked={this.state.muted}
                  onChange={this.handleMuteChange}
                />
                &nbsp;
                Mute graph health alert
              </label>
              <div className="help-block">Check it if you don&apos;t want to see alerts</div>
            </div>
          </div>
        </FormGroup>
      </div>
    );
  }
}

export default connect(
  () => ({
    warningThreshold: window.localStorage.getItem('warningHealthThreshold') || null,
    dangerThreshold: window.localStorage.getItem('dangerHealthThreshold') || null,
    muted: window.localStorage.getItem('alertHeathMuted') === 'true',
  }),
)(AlertingHealthForm);
