import React from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';

import {
  clearAlert, loadAlert, saveAlert, loadAlertTemplate,
} from '../../../store/reducers/alerts/alert';
import { explainAlert, clearAlertExplanation } from '../../../store/reducers/alerts/alertExplanation';

import AlertForm from './AlertForm';
import { parseSearch } from '../../../utils/url';
import { wrapSubProjectEditPage } from '../../projects/wrapSubProjectPage';

class AlertEditPage extends React.Component {
  static getDerivedStateFromProps(props, state) {
    if (!props.alert.templateData) {
      return state;
    }

    return {
      ...state,
      templateId: props.alert.templateData.id,
      templateVersion: props.alert.templateData.templateVersionTag,
    };
  }

  constructor(props) {
    super(props);
    this._projectId = props.match.params.projectId;
    this._alertId = props.match.params.alertId;
    this._isNew = !this._alertId;

    this.state = {};

    const parsedSearch = parseSearch(props.location.search);

    this._isTemplated = (parsedSearch.get('aType') || 'custom') === 'fromTemplate';

    if (this._isNew) {
      this._alertId = parsedSearch.get('duplicateOf');
      this._isFromAutoGraph = props.alert ? props.alert._isFromAutoGraph : false;
    }

    if (this._isTemplated) {
      this.state.templateId = parsedSearch.get('templateId');
      this.state.templateVersion = parsedSearch.get('templateVersion');
    }
  }

  componentDidMount() {
    const { alert } = this.props;
    if (this._alertId) {
      const loadedAlertId = alert.id;
      if (loadedAlertId !== this._alertId) {
        this.props.loadAlert(this._projectId, this._alertId);
      }
    } else if (!this._isFromAutoGraph) {
      this.props.clearAlert();
    }

    if (this._isNew && this._isTemplated && this.state.templateId) {
      this.props.loadAlertTemplate(this.state.templateId, this.state.templateVersion);
    }
  }

  componentWillUnmount() {
    this.props.clearAlert();
    this.props.clearAlertExplanation();
  }

  onSubmitAlert = (alert) => {
    this.props.saveAlert(this._projectId, alert, this._isNew)
      .then((payload) => {
        if (payload) {
          this.props.history.push(`/admin/projects/${this._projectId}/alerts/${payload.id}`);
        }
      });
  };

  get title() {
    return !this._isTemplated ? 'New custom alert' : `New alert from template ${this.state.templateId || ''}`;
  }

  render() {
    const { alert, alertExplanation } = this.props;

    const isEmptyAlert = isEmpty(alert);
    const isExistingEmptyAlert = !this._isNew && isEmptyAlert;
    const isDuplicatedAndEmpty = this._alertId && isEmptyAlert;
    const isFromTemplateWithoutTemplate = (this._isTemplated && !alert.templateData)
      || (!isEmptyAlert && alert.realType?.fromTemplate && !alert.templateData);

    if (isExistingEmptyAlert || isDuplicatedAndEmpty || isFromTemplateWithoutTemplate) {
      return <div>Loading...</div>;
    }

    const isFromTemplate = this._isTemplated || Boolean(alert.templateData);

    return (
      <div>
        <Breadcrumb match={this.props.match} alertName={this._isNew ? '' : alert.name} />
        <PageTitle title={this._isNew ? this.title : `Edit alert ${alert.name}`} />
        <AlertForm
          projectId={this._projectId}
          alert={alert}
          alertExplanation={alertExplanation}
          isNew={this._isNew}
          onSubmit={this.onSubmitAlert}
          onExplain={this.props.explainAlert}
          fromTemplate={isFromTemplate}
          loadAlertTemplate={this.props.loadAlertTemplate}
          isDuplicate={Boolean(this._isNew && this._alertId)}
        />
      </div>
    );
  }
}

AlertEditPage.propTypes = {
  match: PropTypes.object.isRequired,
  location: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  alert: PropTypes.object.isRequired,
  alertExplanation: PropTypes.object.isRequired,
  loadAlert: PropTypes.func.isRequired,
  loadAlertTemplate: PropTypes.func.isRequired,
  explainAlert: PropTypes.func.isRequired,
  saveAlert: PropTypes.func.isRequired,
  clearAlert: PropTypes.func.isRequired,
  clearAlertExplanation: PropTypes.func.isRequired,
};

const mapStateToProps = (state) => ({
  alert: state.alert,
  alertExplanation: state.alertExplanation,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadAlert,
  explainAlert,
  saveAlert,
  clearAlert,
  clearAlertExplanation,
  loadAlertTemplate,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertEditPage);

export default wrapSubProjectEditPage(connectedPage);
