import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import isEmpty from 'lodash/isEmpty';
import entries from 'lodash/entries';
import pick from 'lodash/pick';
import cloneDeep from 'lodash/cloneDeep';
import set from 'lodash/set';
import get from 'lodash/get';

import ApiCallExample from '../../../components/ApiCallExample';
import FormInput from '../../../components/forms/FormInput';
import FormSelect from '../../../components/forms/FormSelect';
import LabelsEditor from '../../../components/LabelsEditor';
import FormRadioGroup from '../../graphs/FormRadioGroup';

import {
  getAlertType,
  ALERT_TYPE_OPTIONS,
  ALERT_TYPES,
  NO_POINTS_POLICY_OPTIONS,
  NO_POINTS_POLICIES,
  RESOLVED_EMPTY_POLICY_OPTIONS,
  RESOLVED_EMPTY_POLICIES,
  DEFAULT_TARGET_STATUS,
  convertAnnotationsListToMap,
  ONLY_TEMPLATE_FIELDS,
  PARAMETERS_FIELDS,
  THRESHOLDS_FIELDS,
  copyFieldsFromTemplate,
} from '../constants';
import FormArea from '../../../components/forms/FormArea';
import FormElement from '../../../components/forms/FormElement';

import { inputTargetValue } from '../../../utils/inputs';
import {
  EditableTable as AnnotationsEditableTable,
  ReadonlyTable as AnnotationsReadOnlyTable,
} from '../components/AnnotationsTable';
import { EditableTable as PredicateRulesEditor } from '../components/PredicateRulesTable';
import AlertFormButtons from './AlertFormButtons';
import AlertExplanation from '../components/AlertExplanation/AlertExplanation';
import ChannelListEditor from './ChannelListEditor/ChannelListEditor';
import FormDurationInput from '../../../components/forms/FormDurationInput';
import TemplateFields from './TemplateFields/TemplateFields';
import { LOAD_ALERT, SERVICE_FIELDS } from '../../../store/reducers/alerts/alert';

class AlertForm extends PureComponent {
  static getParameterValues(state) {
    const result = {};

    PARAMETERS_FIELDS.forEach(([templateName, , , typeName]) => {
      const source = state.templateParameters;

      if (source[templateName]) {
        result[typeName] = Object.values(source[templateName]).map((x) => pick(x, ['name', 'value']));
      }
    });

    THRESHOLDS_FIELDS.forEach(([templateName, , , typeName]) => {
      const source = state.templateThresholds;

      if (source[templateName]) {
        result[typeName] = Object.values(source[templateName]).map((x) => pick(x, ['name', 'value']));
      }
    });

    return result;
  }

  static mapPropsToState({
    alert,
    isNew = false,
    fromTemplate,
    isDuplicate,
  }, fromJsonField) {
    const newAlert = { ...alert };
    delete newAlert._isFromAutoGraph;
    delete newAlert.createdAt;
    delete newAlert.updatedAt;
    delete newAlert.state;

    const newCreated = isNew && !newAlert.id;

    if (isNew) {
      newAlert.id = '';
    }

    if (!newAlert.type) {
      newAlert.type = {};
    }

    const type = getAlertType(newAlert.type, newAlert);

    const templateParameters = {};
    const templateThresholds = {};

    if (fromTemplate && newAlert.templateData) {
      if (isNew) {
        copyFieldsFromTemplate(newAlert, isNew && !fromJsonField);
        newAlert.annotations = (newAlert.annotations || {});
        newAlert.serviceProviderAnnotations = fromJsonField
          ? newAlert.serviceProviderAnnotations
          : (newAlert.templateData.annotations || {});
      }
      const templateFieldsSource = isNew && !isDuplicate
        ? newAlert.templateData
        : newAlert.realType.fromTemplate;

      PARAMETERS_FIELDS.forEach(([defaultName, , , valueName]) => {
        const name = isNew && !isDuplicate ? defaultName : valueName;

        if (!templateFieldsSource[name]) { return; }
        templateParameters[defaultName] = templateParameters[defaultName] || {};

        templateFieldsSource[name].forEach((field) => {
          templateParameters[defaultName][field.name] = {
            ...field,
            value: field.value ?? field.defaultValue,
          };
        });
      });

      THRESHOLDS_FIELDS.forEach(([defaultName, , , valueName]) => {
        const name = isNew && !isDuplicate ? defaultName : valueName;

        if (!templateFieldsSource[name]) { return; }
        templateThresholds[defaultName] = templateThresholds[defaultName] || {};

        templateFieldsSource[name].forEach((field) => {
          templateThresholds[defaultName][field.name] = {
            ...field,
            value: field.value ?? field.defaultValue,
          };
        });
      });

      newAlert.realType = newAlert.type;
      newAlert.type = newAlert.templateData.type;
    }

    const annotationsSource = (
      isNew && fromTemplate && !fromJsonField
        ? newAlert.templateData.annotations : alert.serviceProviderAnnotations
    ) || {};

    const serviceProviderAnnotations = entries(annotationsSource)
      .map((entry) => ({ key: entry[0], value: entry[1] }));

    const annotations = entries(alert.annotations)
      .map((entry) => ({ key: entry[0], value: entry[1] }));

    if (newCreated && !newAlert.windowSecs) {
      newAlert.windowSecs = 300;
    } else if (!newAlert.windowSecs && newAlert.periodMillis) {
      newAlert.windowSecs = Math.trunc(newAlert.periodMillis / 1000);
    }

    if (!newAlert.delaySecs && newAlert.delaySeconds) {
      newAlert.delaySecs = newAlert.delaySeconds;
    }

    delete newAlert.periodMillis;
    delete newAlert.delaySeconds;
    delete newAlert.notificationChannels;

    return {
      alert: newAlert,
      annotations,
      serviceProviderAnnotations,
      type,
      apiVisible: true,
      templateParameters,
      templateThresholds,
      isNew,
      isDuplicate,
      memState: { selected: 0 },
    };
  }

  static updateAlertType(alert, type, update) {
    const alertType = alert.type || {};
    const alertTypeContent = alertType[type] || {};
    const newAlertTypeContent = { ...alertTypeContent, ...update };
    const newAlertType = { ...alertType, [type]: newAlertTypeContent };
    return { ...alert, type: newAlertType };
  }

  static getAlertTypeField(alert, type, field, dflt) {
    const alertType = alert.type || {};
    const alertTypeContent = alertType[type] || {};
    return alertTypeContent[field] || dflt;
  }

  constructor(props) {
    super(props);
    this.state = AlertForm.mapPropsToState(props);
    this._nameInputTouched = !!props.alert.name;
  }

  // TODO: make work with state correctly for React 16
  componentDidUpdate(prevProps) {
    if (this.props.projectId !== prevProps.projectId
      || this.props.alert.templateData !== prevProps.alert.templateData
    ) {
      this.state = AlertForm.mapPropsToState(this.props);
    }
  }

  onInputChange = (event) => {
    const { target } = event;
    const { name } = target;
    const value = inputTargetValue(target);

    const change = {};
    switch (name) {
      case 'id':
        change.id = value;
        if (!this._nameInputTouched && !this.props.fromTemplate) {
          change.name = value;
        }
        break;
      case 'name':
        this._nameInputTouched = true;
        change.name = value;
        break;
      default:
        change[target.name] = value;
    }

    this.setState({ alert: { ...this.state.alert, ...change } });
  };

  onWindowSecsChange = (windowSecs) => {
    this.setState({ alert: { ...this.state.alert, windowSecs } });
  };

  onTypeChange = (event) => {
    this.setState({ type: event.target.value });
  };

  onThresholdInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ alert: AlertForm.updateAlertType(this.state.alert, 'threshold', { [name]: value }) });
  };

  onThresholdRuleDelete = (index) => {
    const rules = [...AlertForm.getAlertTypeField(this.state.alert, 'threshold', 'predicateRules', [])];
    rules.splice(index, 1);

    this.setPredicateRules(this.state.alert, rules);
  };

  onThresholdRuleUpdate = (index, row) => {
    const rules = [...AlertForm.getAlertTypeField(this.state.alert, 'threshold', 'predicateRules', [])];

    if (index < 0) {
      rules.push(row);
    } else {
      rules.splice(index, 1, row);
    }

    this.setPredicateRules(this.state.alert, rules);
  };

  onThresholdRuleMove = (index, direction) => {
    const rules = [...AlertForm.getAlertTypeField(this.state.alert, 'threshold', 'predicateRules', [])];
    const other = index + direction;
    if (other < 0 || other >= rules.length) {
      return;
    }
    [rules[index], rules[other]] = [rules[other], rules[index]];

    this.setPredicateRules(this.state.alert, rules);
  };

  onExpressionInputChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ alert: AlertForm.updateAlertType(this.state.alert, 'expression', { [name]: value }) });
  };

  onNoDataPolicyChange = (event) => {
    if (event.preventDefault) {
      event.preventDefault();
    }

    const { name } = event.target;
    const value = inputTargetValue(event.target);

    this.setState({ alert: { ...this.state.alert, [name]: value } });
  };

  onGroupByLabelsChange = (groupByLabels) => {
    this.setState({ alert: { ...this.state.alert, groupByLabels } });
  };

  onChannelsChange = (channels) => {
    this.setState({ alert: { ...this.state.alert, channels } });
  };

  onNotificationChannelsChange = (notificationChannels) => {
    this.setState({ alert: { ...this.state.alert, notificationChannels } });
  };

  onAnnotationDelete = (index) => {
    const newAnnotationsList = [...this.state.annotations];
    newAnnotationsList.splice(index, 1);

    this.setAnnotations(newAnnotationsList);
  };

  onAnnotationUpdate = (index, row) => {
    const newAnnotationsList = [...this.state.annotations];

    if (index < 0) {
      newAnnotationsList.push(row);
    } else {
      newAnnotationsList.splice(index, 1, row);
    }

    this.setAnnotations(newAnnotationsList);
  };

  onSubmit = (event) => {
    event.preventDefault();
    this.props.onSubmit(this.getCleanAlert());
  };

  onJsonStateChange = (newAlert) => {
    let jsonAlert = newAlert;

    jsonAlert.realType = jsonAlert.type;

    const newAlertTemplateType = newAlert.type?.fromTemplate;

    if (newAlertTemplateType) {
      const [templateId, templateVersionTag] = this.props.isNew
        ? [this.state.alert.templateData.id, this.state.alert.templateData.templateVersionTag]
        : [
          this.state.alert.realType.fromTemplate.templateId,
          this.state.alert.realType.fromTemplate.templateVersionTag,
        ];

      if (
        newAlertTemplateType.templateId !== templateId
        || newAlertTemplateType.templateVersionTag !== templateVersionTag
      ) {
        return this.props.loadAlertTemplate(
          newAlertTemplateType.templateId,
          newAlertTemplateType.templateVersionTag,
        ).then((action) => {
          if (action.type === LOAD_ALERT) {
            jsonAlert = {
              templateData: action.payload.templateData,
              realType: action.payload.templateData.type,
              ...newAlert,
            };
          }
          this.setState(AlertForm.mapPropsToState({
            ...this.props,
            isNew: !this.state.alert.type?.fromTemplate || this.state.isNew,
            alert: jsonAlert,
            fromTemplate: true,
          }, true));
        });
      }

      jsonAlert = {
        templateData: this.props.alert.templateData,
        ...newAlert,
      };
    }

    return this.setState(AlertForm.mapPropsToState({
      ...this.props,
      alert: jsonAlert,
      fromTemplate: this.props.fromTemplate,
    }, true));
  };

  onApiVisibleChange = (apiVisible) => {
    this.setState({ apiVisible });
  };

  onExplainAlert = (event) => {
    event.preventDefault();
    this.props.onExplain(this.props.projectId, this.getCleanAlert());
  };

  setPredicateRules = (alert, rules) => {
    const thresholdAlert = alert.type || {};
    const thresholdAlertContent = thresholdAlert.threshold || {};
    let newThresholdAlertContent = { ...thresholdAlertContent, predicateRules: rules };

    // For compatibility with old gateway
    if (rules.length > 0) {
      newThresholdAlertContent = {
        ...newThresholdAlertContent,
        timeAggregation: rules[0].thresholdType,
        predicate: rules[0].comparison,
        threshold: rules[0].threshold,
      };
    } else {
      delete newThresholdAlertContent.timeAggregation;
      delete newThresholdAlertContent.predicate;
      delete newThresholdAlertContent.threshold;
    }

    const newAlertType = { ...thresholdAlert, threshold: newThresholdAlertContent };
    this.setState({ alert: { ...alert, type: newAlertType } });
  };

  setAnnotations = (annotationsList) => {
    const newAnnotations = convertAnnotationsListToMap(annotationsList);

    this.setState({
      annotations: annotationsList,
      alert: { ...this.state.alert, annotations: newAnnotations },
    });
  };

  getCleanAlert = () => {
    const { type, alert } = this.state;

    const alertType = alert.type || {};

    let typeContent;
    switch (type) {
      case ALERT_TYPES.THRESHOLD: {
        const { threshold } = alertType;
        typeContent = { threshold };
        break;
      }
      case ALERT_TYPES.EXPRESSION: {
        const { expression } = alertType;
        typeContent = { expression };
        break;
      }
      default:
        typeContent = {};
    }

    const result = { ...alert, type: typeContent };

    if (this.props.fromTemplate) {
      if (this.props.isNew) {
        result.type = {
          fromTemplate: {
            templateId: result.templateData.id,
            templateVersionTag: result.templateData.templateVersionTag,
          },
        };
      } else {
        result.type = { ...this.props.alert.type };
      }

      result.type.fromTemplate = {
        ...result.type.fromTemplate,
        ...AlertForm.getParameterValues(this.state),
      };

      ONLY_TEMPLATE_FIELDS.forEach((field) => {
        delete result[Array.isArray(field) ? field[0] : field];
      });
    }

    SERVICE_FIELDS.forEach((field) => {
      delete result[field];
    });

    return result;
  };

  /**
   * @param {'templateParameters' | 'templateThresholds'} fieldKind
   * @param {string} fieldName
   * @param {string} name
   * @param {Function} fn
   */
  updateTemplateData = (fieldKind, fieldName, name, fn = (x) => x) => (
    valOrEvent,
  ) => {
    const value = fn(valOrEvent.target ? valOrEvent.target.value : valOrEvent);
    const newState = cloneDeep(this.state);

    const valuePath = [fieldKind, fieldName, name, 'value'];
    const namePath = [fieldKind, fieldName, name, 'name'];

    set(newState, valuePath, value);
    const hasName = Boolean(get(newState, namePath));
    if (!hasName) {
      set(newState, namePath, name);
    }

    this.setState(newState);
  };

  render() {
    const { projectId, isNew, fromTemplate } = this.props;
    const {
      alert,
      type,
      annotations,
      serviceProviderAnnotations,
      memState,
    } = this.state;

    const alertType = alert.type || {};

    if (fromTemplate && !alert.templateData) {
      return null;
    }

    let typeBlock;
    const actualType = fromTemplate
      ? getAlertType(alert.templateData.type)
      : type;

    switch (actualType) {
      case ALERT_TYPES.THRESHOLD: {
        const threshold = (
          fromTemplate ? alert.templateData.type.threshold : alertType.threshold
        ) || {};

        let predicateRules = threshold.predicateRules || [];
        if (predicateRules.length === 0 && (
          ('timeAggregation' in threshold)
          && ('predicate' in threshold)
          && ('threshold' in threshold)
        )) {
          predicateRules = [{
            thresholdType: threshold.timeAggregation,
            comparison: threshold.predicate,
            threshold: threshold.threshold,
            targetStatus: DEFAULT_TARGET_STATUS,
          }];
        }
        typeBlock = (
          <div>
            <FormElement label="">
              Compare values of a metrics with a user defined threshold
            </FormElement>
            <FormArea
              name="selectors" label="Selectors"
              rows={3}
              disabled={fromTemplate}
              value={threshold.selectors}
              onChange={this.onThresholdInputChange}
              help="Label selectors to define metrics to check"
            />
            <FormElement
              name="predicateRules"
              label="Predicate rules"
              help="Rules that control alert status. The first predicate that is true sets corresponding status. If no predicates are true the status is OK"
            >
              <PredicateRulesEditor
                readOnly={fromTemplate}
                predicateRules={predicateRules}
                onDelete={this.onThresholdRuleDelete}
                onUpdate={this.onThresholdRuleUpdate}
                onMove={this.onThresholdRuleMove}
              />
            </FormElement>
          </div>
        );
        break;
      }
      case ALERT_TYPES.EXPRESSION: {
        const expression = alertType.expression || {};
        typeBlock = (
          <div>
            <FormElement label="">
              Evaluate used defined expression for metric
            </FormElement>
            <FormArea
              name="program" label="Program"
              value={expression.program}
              rows={6}
              disabled={fromTemplate}
              onChange={this.onExpressionInputChange}
              help="Contains expression to evaluate"
            />
            {!fromTemplate && (
              <FormArea
                name="checkExpression" label="Check expression"
                value={expression.checkExpression}
                onChange={this.onExpressionInputChange}
                help="Expression that as a result evaluation should return boolean value,
                  where true it's ALARM, false it's OK. Expression can use variable from program"
              />
            )}
          </div>
        );
        break;
      }
      default:
        typeBlock = (
          <FormElement label="">
            Please select one of alert types!
          </FormElement>
        );
    }

    return (
      <div className="form-horizontal">
        <div className={this.state.apiVisible ? 'col-lg-7 col-md-8' : 'col-lg-10 col-md-11'}>
          <FormInput
            type="text"
            name="id" label="Id"
            value={alert.id}
            onChange={this.onInputChange}
            help="Unique alert identificator"
            disabled={!isNew}
          />
          <FormInput
            type="text"
            name="name" label="Name"
            value={alert.name} onChange={this.onInputChange}
            help="Human-readable name of the alert, e.g.: 'Disk free space'"
          />
          {fromTemplate && (
            <>
              <FormInput
                type="text"
                label="Service Provider"
                value={alert.templateData.serviceProviderId}
                help="Service provider that owns the alert template"
                disabled
              />
              <FormInput
                type="text"
                label="Template"
                help="ID of the template from which the alert is created"
                value={alert.templateData.id}
                disabled
              />
              <FormInput
                type="text"
                label="Template version"
                help="Version of the template from which the alert is created"
                value={alert.templateData.templateVersionTag}
                disabled
              />
            </>
          )}
          <FormArea
            type="text"
            name="description"
            label="Description"
            value={alert.description}
            onChange={this.onInputChange}
            rows={6}
          />
          <FormDurationInput
            name="windowSecs"
            label="Evaluation window"
            format="seconds"
            disabled={fromTemplate}
            value={alert.windowSecs}
            onChange={this.onWindowSecsChange}
            help="Time window which will be used to evaluate the alert"
          />
          <FormInput
            type="number"
            label="Delay"
            suffix="seconds"
            name="delaySecs"
            disabled={fromTemplate}
            value={alert.delaySecs}
            onChange={this.onInputChange}
            help="That much time the alert time window will be shifted to the past. Useful for aggregated metrics, since they are not collected at once."
          />
          {fromTemplate ? (
            <FormInput
              type="text"
              label="Type"
              value={getAlertType(alert.templateData.type)}
              disabled
            />
          ) : (
            <FormRadioGroup
              name="type"
              label="Type"
              value={type}
              defaultValue={ALERT_TYPES.THRESHOLD}
              options={ALERT_TYPE_OPTIONS}
              onChange={this.onTypeChange}
            />
          )}
          {typeBlock}
          {fromTemplate && (
            <TemplateFields
              template={this.props.alert.templateData}
              onUpdate={this.updateTemplateData}
              values={{
                thresholds: this.state.templateThresholds,
                parameters: this.state.templateParameters,
              }}
              isServiceProviderAlert={!this.props.isNew
                && Boolean(alert?.realType?.fromTemplate?.serviceProvider)}
            />
          )}
          <FormSelect
            name="resolvedEmptyPolicy"
            label="No metrics policy"
            value={alert.resolvedEmptyPolicy}
            disabled={fromTemplate}
            defaultValue={RESOLVED_EMPTY_POLICIES.DEFAULT}
            options={RESOLVED_EMPTY_POLICY_OPTIONS}
            onChange={this.onNoDataPolicyChange}
            help="Action that is taken when no metrics are resolved by any of the selectors."
          />
          <FormSelect
            name="noPointsPolicy"
            label="No points policy"
            value={alert.noPointsPolicy}
            disabled={fromTemplate}
            defaultValue={NO_POINTS_POLICIES.DEFAULT}
            options={NO_POINTS_POLICY_OPTIONS}
            onChange={this.onNoDataPolicyChange}
            help="Action that is taken when there are timeseries with no points in alert window."
          />
          <FormElement
            label="Group by labels"
            help="List of label key that should be use to group metrics, each group
              it's separate sub alert that check independently from other group"
          >
            <LabelsEditor
              values={alert.groupByLabels}
              onChange={this.onGroupByLabelsChange}
              readOnly={fromTemplate}
            />
          </FormElement>
          <FormElement
            label="Annotations"
            help="Templates that explain alert evaluation status, and what to do when the alert is triggered. Which set of variables is available in templates depends on the alert type"
          >
            <AnnotationsEditableTable
              annotations={annotations}
              onDelete={this.onAnnotationDelete}
              onUpdate={this.onAnnotationUpdate}
            />
          </FormElement>
          {fromTemplate && !isEmpty(serviceProviderAnnotations) && (
            <FormElement
              label="Service provider annotations"
              help="Templates provided by service provider that explain alert evaluation status, and what to do when the alert is triggered. Which set of variables is available in templates depends on the alert type"
            >
              <AnnotationsReadOnlyTable
                annotations={serviceProviderAnnotations}
              />
            </FormElement>
          )}
          <FormElement
            label="Channels"
            help={(
              <p>
                Notification channels that will be used to report the alert evaluation status.
                <br />
                You can click
                {' '}
                <a href={`/admin/projects/${projectId}/channels/new`} target="_blank" rel="noopener noreferrer">here</a>
                {' '}
                to create new channel or click to any selected channel to edit it.
              </p>
            )}
          >
            <ChannelListEditor
              projectId={projectId}
              channels={alert.channels || []}
              onChange={this.onChannelsChange}
            />
          </FormElement>
          {!isEmpty(this.props.alertExplanation) && (
            <FormElement label="Explanation">
              <AlertExplanation
                explanation={this.props.alertExplanation}
                showStatusLabel
                showScalars
                memState={memState}
              />
            </FormElement>
          )}
          <AlertFormButtons onSubmit={this.onSubmit} onExplain={this.onExplainAlert} />
        </div>
        <div className={this.state.apiVisible ? 'col-lg-5 col-md-4' : 'col-lg-1 col-md-2'}>
          <ApiCallExample
            path={`/projects/${projectId}/alerts`}
            value={this.getCleanAlert()}
            isNew={this.props.isNew}
            onChange={this.onJsonStateChange}
            visible={this.state.apiVisible}
            onChangeVisibility={this.onApiVisibleChange}
          />
        </div>
      </div>
    );
  }
}

AlertForm.propTypes = {
  projectId: PropTypes.string.isRequired,
  alert: PropTypes.object.isRequired,
  alertExplanation: PropTypes.object.isRequired,
  isNew: PropTypes.bool.isRequired,
  isDuplicate: PropTypes.bool.isRequired, // eslint-disable-line react/no-unused-prop-types
  onSubmit: PropTypes.func.isRequired,
  onExplain: PropTypes.func.isRequired,
  fromTemplate: PropTypes.bool.isRequired,
  loadAlertTemplate: PropTypes.func.isRequired,
};

export default AlertForm;
