import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { formatDurationSecs, parseDurationSecs } from '../../../../utils/duration';

const OPTIONS = [
  { value: '5m', title: '5 minutes' },
  { value: '10m', title: '10 minutes' },
  { value: '30m', title: '30 minutes' },
  { value: '1h', title: '1 hour' },
];

export default class ExtendedDurationInput extends PureComponent {
  constructor(props) {
    super(props);

    this.state = { value: formatDurationSecs(props.value), isValid: true, editMode: false };
  }

  componentDidUpdate(prevProps) {
    if (this.props.value !== prevProps.value) {
      this.setState({ value: formatDurationSecs(this.props.value), isValid: true });
    }
  }

  handleGoToEditMode = () => {
    const seconds = parseDurationSecs(this.state.value);
    const isValid = seconds !== null;
    this.setState({ editMode: true, isValid });
  };

  handleGoToSelectMode = () => {
    this.setState((state) => ({ editMode: false, isValid: true, value: (state.isValid ? state.value : '0') }));
  };

  handleInputChange = (event) => {
    event.preventDefault();
    const { value } = event.target;
    this.handleValueChange(value);
  };

  handleSelect = (event) => {
    event.preventDefault();
    const { value } = event.target;
    if (value === 'custom') {
      this.handleGoToEditMode();
    } else {
      this.handleValueChange(value);
    }
  };

  handleValueChange = (value) => {
    const seconds = parseDurationSecs(value);
    const isValid = seconds !== null;
    this.setState({ value, isValid });
    if (isValid) {
      this.props.onChange(seconds);
    }
  };

  renderSelect(value, name) {
    const isNever = !value || value === '0';

    let isCustom;
    if (isNever) {
      isCustom = false;
    } else {
      isCustom = true;
      for (let i = 0; i < OPTIONS.length; ++i) {
        if (OPTIONS[i].value === value) {
          isCustom = false;
          break;
        }
      }
    }

    let selectedValue;
    if (isNever) {
      selectedValue = '0';
    } else if (isCustom) {
      selectedValue = 'custom';
    } else {
      selectedValue = value;
    }

    return (
      <div>
        <select
          className="form-control"
          style={{ minWidth: '100px' }}
          name={name}
          value={selectedValue}
          onChange={this.handleSelect}
        >
          <option key="never" value="0">Never</option>
          {OPTIONS.map((option) => (
            <option key={option.value} value={option.value}>
              {option.title}
            </option>
          ))}
          <option key="custom" value="custom">
            Custom
            {isCustom ? ` (${value})` : ''}
          </option>
        </select>
      </div>
    );
  }

  renderCustomEditor(name, value) {
    return (
      <div className={this.state.isValid ? '' : ' has-error'}>
        <input
          className="form-control"
          name={name}
          value={value}
          onChange={this.handleInputChange}
        />
        <div className="help-block">
          Go to
          {' '}
          <a onClick={this.handleGoToSelectMode}>select</a>
        </div>
      </div>
    );
  }

  render() {
    const { name } = this.props;
    const { value, editMode } = this.state;

    if (editMode) {
      return this.renderCustomEditor(name, value);
    }

    return this.renderSelect(value, name);
  }
}

ExtendedDurationInput.propTypes = {
  name: PropTypes.string.isRequired,
  value: PropTypes.number,
  onChange: PropTypes.func.isRequired,
};

ExtendedDurationInput.defaultProps = {
  value: null,
};
