import PropTypes from 'prop-types';
import React, { PureComponent } from 'react';

import FormInput from '../../../../components/forms/FormInput';
import LabelsEditor from '../../../../components/LabelsEditor';

import {
  PARAMETERS_FIELDS,
  THRESHOLDS_FIELDS,
} from '../../constants';
import FormElement from '../../../../components/forms/FormElement';
import FormGroup from '../../../../components/forms/FormGroup';

export default class TemplateFields extends PureComponent {
  render() {
    const {
      template, onUpdate, values, isServiceProviderAlert,
    } = this.props;

    const hasAnyParameter = PARAMETERS_FIELDS.some(([name]) => Boolean(template[name]));
    const hasAnyThreshold = THRESHOLDS_FIELDS.some(([name]) => Boolean(template[name]));

    const parameters = hasAnyParameter && (
      <FormGroup title="Parameters">
        {PARAMETERS_FIELDS.map(([field, type, inputType]) => {
          if (!template[field]) {
            return null;
          }

          const tFields = template[field];

          if (type === 'text') {
            return tFields.map((tField) => (
              <FormInput
                key={tField.name}
                disabled={isServiceProviderAlert}
                type={inputType}
                name={tField.name}
                label={tField.title}
                value={values.parameters[field][tField.name].value}
                onChange={onUpdate('templateParameters', field, tField.name, inputType === 'number' ? Number : String)}
              />
            ));
          }

          return tFields.map((tField) => {
            const fieldData = values.parameters[field];
            if (!fieldData) { return null; }
            const value = fieldData[tField.name].value;
            return (
              <FormElement label={tField.title} key={tField.name}>
                <LabelsEditor
                  readOnly={isServiceProviderAlert}
                  additional={{ noBadge: inputType === 'text' }}
                  values={value}
                  onChange={onUpdate('templateParameters', field, tField.name)}
                />
              </FormElement>
            );
          });
        })}
      </FormGroup>
    );

    const threshold = hasAnyThreshold && (
      <FormGroup title="Thresholds">
          {THRESHOLDS_FIELDS.map(([field, type, inputType]) => {
            if (!template[field]) {
              return null;
            }

            const tFields = template[field];

            if (type === 'text') {
              return tFields.map((tField) => {
                const iField = values.thresholds?.[field]?.[tField.name]
                  || template[field]?.find((th) => th.name === tField.name);

                if (iField === undefined || iField === null) {
                  return null;
                }

                return (
                  <FormInput
                    key={tField.name}
                    type={inputType}
                    name={tField.name}
                    label={tField.title}
                    value={iField.value || iField.defaultValue}
                    onChange={onUpdate('templateThresholds', field, tField.name, inputType === 'number' ? Number : String)}
                  />
                );
              });
            }

            return tFields.map((tField) => {
              const fieldData = values.thresholds[field];
              if (!fieldData) { return null; }
              const value = fieldData[tField.name].value;
              return (
                <FormElement label={tField.title} key={tField.name}>
                  <LabelsEditor
                    additional={{ noBadge: inputType === 'text' }}
                    values={value}
                    onChange={onUpdate('templateThresholds', field, tField.name)}
                  />
                </FormElement>
              );
            });
          })}
      </FormGroup>
    );

    return (
      <>
        {parameters}
        {threshold}
        <hr />
      </>
    );
  }
}

TemplateFields.propTypes = {
  template: PropTypes.object.isRequired,
  onUpdate: PropTypes.object.isRequired,
  values: PropTypes.shape({
    parameters: PropTypes.object,
    thresholds: PropTypes.object,
  }).isRequired,
  isServiceProviderAlert: PropTypes.bool.isRequired,
};
