import React, { PureComponent } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { bindActionCreators } from 'redux';
import isEmpty from 'lodash/isEmpty';

import Breadcrumb from '../../../components/Breadcrumb/Breadcrumb';
import PageTitle from '../../../components/PageTitle';
import ShowActions from '../../../components/ShowActions';
import AlertEvalInfo from '../components/AlertEvalInfo';
import AlertNotificationInfo from '../components/AlertNotificationInfo';
import MultiAlertStatusChart from '../charts/MultiAlertStatusChart';
import SingleAlertStatus from '../charts/SingleAlertStatus/SingleAlertStatus';

import AlertMutesPanel from './AlertMutesPanel';
import {
  loadAlertInterpolated as loadAlert, deleteAlert, changeAlertState,
} from '../../../store/reducers/alerts/alert';
import { loadAlertEvalState } from '../../../store/reducers/alerts/alertEvalState';
import { loadAlertEvalStats } from '../../../store/reducers/alerts/alertEvalStats';
import { loadAlertNotificationState } from '../../../store/reducers/alerts/alertNotificationState';
import { loadAlertNotificationStats } from '../../../store/reducers/alerts/alertNotificationStats';
import { loadExistingAlertExplanation } from '../../../store/reducers/alerts/existingAlertExplanation';

import { wrapSubProjectReadPage } from '../../projects/wrapSubProjectPage';
import { PROJECT_PERMISSIONS } from '../../../auth/ProjectPermissions';
import GenericOptionsPanel from './GenericOptionsPanel';
import MultiAlertStats from './MultiAlertStatsBlock/MultiAlertStatsBlock';

class AlertPage extends PureComponent {
  componentDidMount() {
    const { match } = this.props;
    const { projectId, alertId } = match.params;
    this.fetchAlertWithState(projectId, alertId);
  }

  componentDidUpdate(prevProps) {
    const { projectId, alertId } = this.props.match.params;
    const prevId = prevProps.match.params.alertId;
    if (alertId !== prevId) {
      this.fetchAlertWithState(projectId, alertId);
    }
  }

  onDeleteAlert = (event) => {
    event.preventDefault();
    const { projectId, alertId } = this.props.match.params;
    this.props.deleteAlert(projectId, alertId)
      .then((action) => {
        if (action) {
          this.props.history.push(`/admin/projects/${projectId}/alerts`);
        }
      });
  };

  onDeactivateAlert = (event) => {
    event.preventDefault();
    const { projectId } = this.props.match.params;
    this.props.changeAlertState(projectId, this.props.alert, 'MUTED');
  };

  onActivateAlert = (event) => {
    event.preventDefault();
    const { projectId } = this.props.match.params;
    this.props.changeAlertState(projectId, this.props.alert, 'ACTIVE');
  };

  onMuteAlert = (event) => {
    event.preventDefault();
    const { projectId } = this.props.match.params;
    const { alert } = this.props;
    this.props.history.push(`/admin/projects/${projectId}/mutes/new?alertId=${alert.id}`);
  }

  onLoadExplanation = (ts) => {
    const { projectId, alertId } = this.props.match.params;
    const time = new Date(ts).toISOString();
    return this.props.loadExistingAlertExplanation(projectId, alertId, { time });
  };

  fetchAlertWithState(projectId, alertId) {
    this.props.loadAlert(projectId, alertId)
      .then((action) => {
        if (action) {
          const alert = action.payload;
          if (isEmpty(alert.groupByLabels)) {
            this.props.loadAlertEvalState(projectId, alertId);
            this.props.loadAlertNotificationState(projectId, alertId);
            this.props.loadExistingAlertExplanation(projectId, alertId);
          } else {
            this.props.loadAlertEvalStats(projectId, alertId);
            this.props.loadAlertNotificationStats(projectId, alertId);
          }
        }
      });
  }

  render() {
    const {
      alert,
      evalState, notificationState,
      evalStats, notificationStats,
      alertExplanation,
      match,
      projectAuth,
    } = this.props;

    const { projectId, alertId } = match.params;

    if (isEmpty(alert) || (alert.type.fromTemplate && !alert.templateData)) {
      return <span>Loading...</span>;
    }

    const isMultiAlert = !isEmpty(alert.groupByLabels);

    const isActive = alert.state === 'ACTIVE';

    const genericOptionsPanel = <GenericOptionsPanel alert={alert} />;

    let statusBlock;
    let statusChart;

    if (isMultiAlert) {
      statusBlock = (
        <MultiAlertStats
          projectId={projectId}
          alertId={alertId}
          evalStats={evalStats}
          notificationStats={notificationStats}
        />
      );

      statusChart = (
        <MultiAlertStatusChart projectId={projectId} alertId={alertId} />
      );
    } else {
      statusBlock = (
        <div>
          <AlertEvalInfo
            projectId={projectId}
            alertId={alertId}
            evalState={evalState}
          />
          <AlertNotificationInfo projectId={projectId} notificationState={notificationState} />
        </div>
      );

      statusChart = (
        <SingleAlertStatus
          projectId={projectId}
          alertId={alertId}
          explanation={alertExplanation}
          loadExplanation={this.onLoadExplanation}
        />
      );
    }

    const title = `Alert ${alert.name}`;

    let subTitle;

    if (isActive) {
      subTitle = '';
    } else {
      subTitle = (<span className="label label-warning">Muted</span>);
    }

    const canEdit = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_UPDATE);
    const canDelete = projectAuth.isAuthorizedFor(PROJECT_PERMISSIONS.CONFIG_DELETE);

    const showActionsBar = (
      <ShowActions
        editHref={`/admin/projects/${projectId}/alerts/${alertId}/edit`}
        duplicatePath={`/admin/projects/${projectId}/alerts/new?duplicateOf=${alertId}`}
        mutePath={`/admin/projects/${projectId}/mutes/new?alertId=${alertId}`}
        canMute
        onActivate={this.onActivateAlert}
        onDeactivate={this.onDeactivateAlert}
        onMute={this.onMuteAlert}
        onDelete={this.onDeleteAlert}
        canEdit={canEdit}
        canChangeState={canEdit}
        canDelete={canDelete}
        isActive={isActive}
      />
    );

    return (
      <div>
        <Breadcrumb match={this.props.match} alertName={alert.name} />
        <PageTitle title={title} subTitle={subTitle} />
        <div className="row">
          <div className="col-lg-12">
            {statusChart}
          </div>
        </div>
        <div className="row">
          <div className={isMultiAlert ? 'col-lg-9' : 'col-lg-8'}>
            {genericOptionsPanel}
            {showActionsBar}
          </div>
          <div className={isMultiAlert ? 'col-lg-3' : 'col-lg-4'}>
            {statusBlock}
            <AlertMutesPanel evalState={evalState} projectId={alert.projectId} />
          </div>
        </div>
      </div>
    );
  }
}

AlertPage.propTypes = {
  match: PropTypes.object.isRequired,
  history: PropTypes.object.isRequired,
  alert: PropTypes.object.isRequired,
  evalState: PropTypes.object.isRequired,
  evalStats: PropTypes.object.isRequired,
  notificationState: PropTypes.object.isRequired,
  notificationStats: PropTypes.object.isRequired,
  alertExplanation: PropTypes.object.isRequired,
  loadAlert: PropTypes.func.isRequired,
  loadAlertEvalState: PropTypes.func.isRequired,
  loadAlertEvalStats: PropTypes.func.isRequired,
  loadAlertNotificationState: PropTypes.func.isRequired,
  loadExistingAlertExplanation: PropTypes.func.isRequired,
  loadAlertNotificationStats: PropTypes.func.isRequired,
  changeAlertState: PropTypes.func.isRequired,
  deleteAlert: PropTypes.func.isRequired,
  projectAuth: PropTypes.object.isRequired,
};

const mapStateToProps = (state) => ({
  alert: state.alert,
  evalState: state.alertEvalState,
  evalStats: state.alertEvalStats,
  alertExplanation: state.existingAlertExplanation,
  notificationState: state.alertNotificationState,
  notificationStats: state.alertNotificationStats,
  subAlerts: state.subAlerts,
});

const mapDispatchToProps = (dispatch) => bindActionCreators({
  loadAlert,
  loadAlertEvalState,
  loadAlertEvalStats,
  loadAlertNotificationState,
  loadAlertNotificationStats,
  loadExistingAlertExplanation,
  deleteAlert,
  changeAlertState,
}, dispatch);

const connectedPage = connect(mapStateToProps, mapDispatchToProps)(AlertPage);

export default wrapSubProjectReadPage(connectedPage);
